function model = aero();
  
# --------------------------------------------------------------------
# Example: Frequency response analysis of a standard class glider
#          Aerodynamic model
# --------------------------------------------------------------------

# ====================================================================
# Data (kg, m, s)
# ====================================================================

# Geometry

  geodata

# Discretization

  nsa =   5;      % No. of airfoil spline segments
  nxi =  50;      % No. of panels in x-direction of inner wing
  nxo =  40;      % No. of panels in x-direction of outer wing
  nxa =  10;      % No. of panels in x-direction of aileron
  nyi =  20;      % No. of panels in y-direction of inner wing
  nyo =  24;      % No. of panels in y-direction of outer wing

# Environment and Manoeuvres

  v     =   30;   % Flight velocity
  rho   = 1.21;   % Mass density of air
  g     = 9.81;   % Gravity acceleration

# ====================================================================
# Model Definition
# ====================================================================

  model = struct("type", "aero", "subtype", "vlm", "cref", cr);

  yt  = 0.5 * b;           % y-coordinate of wing tip
  ya  = (1 - lr) * yt;     % y-coordinate of aileron

  dihedral = tand(delta);
  zt = yt * dihedral;      % Wing tip
  za = ya * dihedral;      % Begin of aileron

  ca1 = fr * cr;           % inner aileron chord length
  ca2 = fr * ct;           % outer aileron chord length
  co1 = cr - ca1;          % inner outer wing chord length
  co2 = ct - ca2;          % outer outer wing chord length

# Right wing
# ----------

  points = struct("id",   {1, 2, 3, 4, 5},
                  "coor", {[xle, 0, 0], ...
                           [xle, ya, za], ...
                           [xle, yt, zt], ...
                           [xle + co1, ya, za], ...
                           [xle + co2, yt, zt]});

# Inner wing, outer wing, aileron

  lsf(1 : 3) = struct("id",     {10, 11, 12},                   
                      "points", {[1, 2], [2, 3], [4, 5]},
                      "chord",  {cr, [co1, co2], [ca1, ca2]},
                      "nx",     {nxi, nxo, nxa},
                      "ny",     {nyi, nyo, nyo},
                      "typex",  "linear",
                      "typey",  {"linear", "cos>", "cos>"});

# -------------------------------------------------------------------

# Left wing
# ---------

  points(6 : 10) = struct("id",   {6, 7, 8, 9, 10},
                          "coor", {[xle, 0, 0], ...
                                   [xle, -ya, za], ...
                                   [xle, -yt, zt], ...
                                   [xle + co1, -ya, za], ...
                                   [xle + co2, -yt, zt]});

# Inner wing, outer wing, aileron

  lsf(4 : 6) = struct("id",     {20, 21, 22},
                      "points", {[7, 6], [8, 7], [10, 9]},
                      "chord",  {cr, [co2, co1], [ca2, ca1]},
                      "nx",     {nxi, nxo, nxa},
                      "ny",     {nyi, nyo, nyo},
                      "typex",  "linear",
                      "typey",  {"linear", "<cos", "<cos"});

# -------------------------------------------------------------------

# Vertical stabilizer
# -------------------

  points(11 : 14) = struct("id",   {31, 32, 33, 34},
                           "coor", {[l - 0.6, 0, 0], ...
                                    [l - 0.5, 0, h], ...
                                    [l, 0, 0], ...
                                    [l, 0, h]});
# Fin and rudder

  lsf(7 : 8) = struct("id",     {31, 32},
                      "points", {[31, 32], [33, 34]},
                      "chord",  {[0.6, 0.5], 0.2},
                      "nx",     {30, 10},
                      "ny",     { 8,  8},
                      "typex",  "linear",
                      "typey",  "cos");

# -------------------------------------------------------------------

# Horizontal stabilizer
# ---------------------

  points(15 : 18) = struct("id",   {41, 42, 43, 44},
                           "coor", {[l - 0.4, 0.5 * be, h], ...
                                    [l, 0.5 * be, h], ...
                                    [l - 0.4, -0.5 * be, h], ...
                                    [l, -0.5 * be, h]});

# Fin

  lsf(9 : 10) = struct("id",     {41, 42},
                       "points", {[32, 41], [43, 32]},
                       "chord",  {[0.5, 0.4], [0.4, 0.5]},
                       "nx",     30,
                       "ny",      8,
                       "typex",  "linear",
                       "typey",  {"cos>", "<cos"});

# Elevator

  lsf(11 : 12) = struct("id",     {43, 44},
                        "points", {[34, 42], [44, 34]},
                        "chord",  0.1, 
                        "nx",     6,
                        "ny",     8,
                        "typex",  "linear",
                        "typey",  {"cos>", "<cos"});

# -------------------------------------------------------------------

  model.points = points;
  model.ls     = lsf;

# Control surfaces
# ----------------

# Aileron  (positive: stick to the right)
# Rudder   (positive: right pedal)
# Elevator (positive: pull)

  controls = struct("name",    {"aileron", "rudder", "elevator"},
                    "ls",      { [12, 22],    32,     [43, 44]},
                    "factors", { [-1, 1],      1,     [-1, -1]});

  model.controls = controls;

endfunction
