# Example: Gull wing
#
# Splines
#  - Definition of aeroelastic model
#  - Transfer of aerodynamic loads to solid model
#  - Computation of displacements
#  - Storage of aeroelastic component in binary file aeroelastic.bin
#
# Files needed: solid.bin, aero.bin
#
# ---------------------------------------------------------------------

  fid = fopen("splines.res", "wt");

# Data
# ----

  nsi = 8;    % Number of spline breaks of outer wing
  nso = 4;    % Number of spline breaks of inner wing

# Aeroelastic model
# ------------------

  load solid.bin
  load aero.bin

  model = struct("type", "aeroelastic",
                 "solid", wings, "aero", winga);

# Splines

  model.splines = struct("id",    {1, 2, 3},
                         "type",  "tb",
                         "lsid",  {1, 2, 3},
                         "nodes", {"inner_wing", ...
                                   "outer_wing", ...
                                   "outer_wing"},
                         "data",  {struct("nbreaks", nsi), ...
                                   struct("nbreaks", nso), ...
                                   struct("nbreaks", nso)});

# Analysis
# --------

# Create component and compute splines

  wing = mfs_new(fid, model);
  wing = mfs_splines(wing);

  save -binary aeroelastic.bin wing rho

# Transfer aerodynamic loads from rigid aerodynamic analysis
# to the solid component and retrieve load resultants

  wings = mfs_transfer(wing, winga, "statresp", "loads");
  [FA, MA] = mfs_getresp(winga, "statresp", "aeload");
  [FS, MS] = mfs_getresp(wings, "load", "resultant");
  
  nc = columns(FA); 

  fprintf(fid, "Load resultants of aerodynamic component: \n\n");
  for n = 1 : nc
      fprintf(fid, "  Configuration %2d:\n", n)
      fprintf(fid, "     F = [%10.3e, %10.3e, %10.3e] kN\n",
              FA(:, n) / 1000);
      fprintf(fid, "     M = [%10.3e, %10.3e, %10.3e] kNm\n",
              MA(:, n) * 1e-6);
  endfor

  fprintf(fid, "\nLoad resultants of solid component: \n\n");
  for n = 1 : nc
      fprintf(fid, "  Configuration %2d:\n", n)
      fprintf(fid, "     F = [%10.3e, %10.3e, %10.3e] kN\n",
              FS(:, n) / 1000);
      fprintf(fid, "     M = [%10.3e, %10.3e, %10.3e] kNm\n",
              MS(:, n) * 1e-6);
  endfor

# Compute displacements and transfer them to the aerodynamic
# component

  wings = mfs_statresp(wings);
  winga = mfs_transfer(wing, wings, "statresp", "disp");

  mfs_print(fid, wings, "statresp", "reac");

  mfs_export("solid.dsp", "msh", wings, "statresp", "disp");
  mfs_export("aero.dsp", "msh", winga, "statresp", "disp");

# Combine the models and the displacements

  mfs_merge("solid.msh", "aero.msh", "wing.msh", "msh");
  mfs_merge("solid.dsp", "aero.dsp", "wing.dsp", "msh");

  fclose(fid);
