/* --------------------------------------------------------------------

   Example: Swept Wing

   Units: mm

   Physical Groups:

     Ribs_Upper_Part     Line     Upper par of ribs
     Ribs_Lower_Part     Line     Lower part of ribs
     Ribs_Front_Part     Line     Front part or ribs
     Ribs_End_Part       Line     Rear part of ribs

     Ribs                Surface  Rib membrane

     Stringer_1_Upper    Line     1st upper stringer
     Stringer_2_Upper    Line     2nd upper stringer
     Stringer_3_Upper    Line     3rd upper stringer
     Stringer_4_Upper    Line     4th upper stringer
     Stringer_1_Lower    Line     1st lower stringer
     Stringer_2_Lower    Line     2nd lower stringer
     Stringer_3_Lower    Line     3rd lower stringer
     Stringer_4_Lower    Line     4th lower stringer

     Front_Spar          Surface  Front spar membrane
     Main_Spar           Surface  Main spar membrane
     Back_Spar           Surface  Back spar membrane
     Rear_Spar           Surface  Rear spar membran

     Skin                Surface  Skin membrane

     Clamped             Points   Constraints

-------------------------------------------------------------------- */

  DefineConstant [ prt = 0 ];
  DefineConstant [ prtfil = "solid.prt" ];

  If (prt == 1)
     Printf("Model Information") > prtfil;
  EndIf

/* --------------------------------------------------------------------
   Data
-------------------------------------------------------------------- */

  Include "constants.geo";

  If (prt == 1)
     Printf("Constants:") >> prtfil;
     Printf("  b      = %f", b) >> prtfil;
     Printf("  sweep  = %f", sweep) >> prtfil;
     Printf("  c_root = %f", c_root) >> prtfil;
     Printf("  c_tip  = %f", c_tip) >> prtfil;
  EndIf

/* --------------------------------------------------------------------
   Airfoil coordinates relative to chord length (NACA 63(2)-615)
-------------------------------------------------------------------- */

   x_1 = 0.10;
   x_2 = 0.25;
   x_3 = 0.55;
   x_4 = 0.80;

   z_1 = 6.6E-2;
   z_2 = 7.6E-2;
   z_3 = 5.9E-2;
   z_4 = 3.1E-2;

   z_5 = -3.0E-2;
   z_6 = -4.3E-2;
   z_7 = -3.9E-2;
   z_8 = -2.2E-2;

/* --------------------------------------------------------------------
   Points
-------------------------------------------------------------------- */

   swp = Tan(sweep * Pi / 180);
   dy  = b / 8;

   For k In {1 : 9}
       y  = (k - 1) * dy;
       s  = y / b;
       c  = (1 - s) * c_root + s * c_tip;
       xO = swp * y - 0.25 * c;
       If (prt == 1)
          Printf("y = %f, c = %f, xO = %f", y, c, xO) >> prtfil;
       EndIf
       For l In {1 : 4}
          P_WS~{k}~{l} = newp;
          Point(P_WS~{k}~{l}) = (xO + x~{l} * c, y, z~{l} * c);
          m = l + 4;
          P_WS~{k}~{m} = newp;
          Point(P_WS~{k}~{m}) = (xO + x~{l} * c, y, z~{m} * c);
          If (prt == 1)
             h = (z~{l} - z~{m}) * c;
             Printf("h%1.0f = %f", l, h) >> prtfil;
          EndIf
       EndFor
   EndFor

/* --------------------------------------------------------------------
   Ribs
-------------------------------------------------------------------- */

// Upper Part (Identifiers 1 to 3)

   For k In {1 : 9}
       For l In {1 : 3}
           R_WS~{k}~{l} = newl;
           Line(R_WS~{k}~{l}) = {P_WS~{k}~{l}, P_WS~{k}~{l+1}};
       EndFor
   EndFor

   Physical Line("Ribs_Upper_Part") = {R_WS_1_1 : R_WS_9_3};

// Lower Part (Identifiers 4 to 6)

   For k In {1 : 9}
       For l In {4 : 6}
           R_WS~{k}~{l} = newl;
           Line(R_WS~{k}~{l}) = {P_WS~{k}~{l+1}, P_WS~{k}~{l+2}};
       EndFor
   EndFor

   Physical Line("Ribs_Lower_Part") = {R_WS_1_4 : R_WS_9_6};

// Vertical Parts (Identifiers 7 to 10)

   For k In {1 : 9}
       For l In {1 : 4}
           R_WS~{k}~{l+6} = newl;
           Line(R_WS~{k}~{l+6}) = {P_WS~{k}~{l+4}, P_WS~{k}~{l}};
           v~{l}[k-1] = R_WS~{k}~{l+6};
       EndFor
   EndFor

   Physical Line("Ribs_Front_Part") = v_1[];
   Physical Line("Ribs_End_Part") = v_4[];

/* --------------------------------------------------------------------
   Stringers
-------------------------------------------------------------------- */

// Upper stringers (SU) and lower stringers (SL)

   For j In {1 : 4}
       For k In {1 : 8}
           SU~{j}~{k} = newl;
           Line(SU~{j}~{k}) = {P_WS~{k}~{j}, P_WS~{k+1}~{j}};
           su~{j}[k-1] = SU~{j}~{k};
           SL~{j}~{k} = newl;
           Line(SL~{j}~{k}) = {P_WS~{k}~{j+4}, P_WS~{k+1}~{j+4}};
           sl~{j}[k-1] = SL~{j}~{k};
       EndFor
   EndFor

   Physical Line("Stringer_1_Upper") = su_1[];
   Physical Line("Stringer_2_Upper") = su_2[];
   Physical Line("Stringer_3_Upper") = su_3[];
   Physical Line("Stringer_4_Upper") = su_4[];

   Physical Line("Stringer_1_Lower") = sl_1[];
   Physical Line("Stringer_2_Lower") = sl_2[];
   Physical Line("Stringer_3_Lower") = sl_3[];
   Physical Line("Stringer_4_Lower") = sl_4[];

/* --------------------------------------------------------------------
   Shear Panels
-------------------------------------------------------------------- */

// Front Spar

   For k In {1 : 8}
       lopno = newll;
       Line Loop(lopno) = 
            {SU~{1}~{k}, -R_WS~{k+1}~{7}, -SL~{1}~{k}, R_WS~{k}~{7}};
       FS~{k} = news;
       Plane Surface(FS~{k}) = {lopno};
       FS[k-1] = FS~{k};
   EndFor

  Physical Surface("Front_Spar") = {FS[]};

// Main Spar

   For k In {1 : 8}
       lopno = newll;
       Line Loop(lopno) = 
            {SU~{2}~{k}, -R_WS~{k+1}~{8}, -SL~{2}~{k}, R_WS~{k}~{8}};
       MS~{k} = news;
       Plane Surface(MS~{k}) = {lopno};
       MS[k-1] = MS~{k};
   EndFor

   Physical Surface("Main_Spar") = {MS[]};

// Back Spar

   For k In {1 : 8}
       lopno = newll;
       Line Loop(lopno) = 
            {SU~{3}~{k}, -R_WS~{k+1}~{9}, -SL~{3}~{k}, R_WS~{k}~{9}};
       BS~{k} = news;
       Plane Surface(BS~{k}) = {lopno};
       BS[k-1] = BS~{k};
   EndFor

   Physical Surface("Back_Spar") = {BS[]};

// Rear Spar

   For k In {1 : 8}
       lopno = newll;
       Line Loop(lopno) = 
            {SU~{4}~{k}, -R_WS~{k+1}~{10}, -SL~{4}~{k}, R_WS~{k}~{10}};
       RS~{k} = news;
       Plane Surface(RS~{k}) = {lopno};
       RS[k-1] = RS~{k};
   EndFor

   Physical Surface("Rear_Spar") = {RS []};

// Ribs

   nr = 0;

   For k In {1 : 9}
       For l In {0 : 2}
           lopno = newl;
           Line Loop(lopno) = {R_WS~{k}~{4+l}, R_WS~{k}~{8+l},
                              -R_WS~{k}~{1+l}, -R_WS~{k}~{7+l}};
           Ribs[nr] = news;
           Plane Surface(Ribs[nr]) = {lopno};
           nr++;
       EndFor
   EndFor

   Physical Surface("Ribs") = Ribs[];

// Skin

   sk = 0;
   For k In {1 : 8}
       For j In {1 : 3}
           lopno = newll;
           Line Loop(lopno) = {R_WS~{k}~{j}, SU~{j+1}~{k}, 
                               -R_WS~{k+1}~{j}, -SU~{j}~{k}};
           Skin[sk] = news;
           Plane Surface(Skin[sk]) = {lopno};
           sk++;
           lopno = newll;
           Line Loop(lopno) = {R_WS~{k}~{j+3}, SL~{j+1}~{k}, 
                               -R_WS~{k+1}~{j+3}, -SL~{j}~{k}};
           Skin[sk] = news;
           Plane Surface(Skin[sk]) = {lopno};
           sk++;
       EndFor 
   EndFor 

   Physical Surface("Skin") = Skin[];

/* --------------------------------------------------------------------
   Constraints
-------------------------------------------------------------------- */

   Physical Point("Clamped") = {P_WS_1_2, P_WS_1_3, P_WS_1_6, P_WS_1_7};

   fontsize = 24;
   fonttype =  4;
   textpos  =  1;
   font = fontsize + 2^8 * fonttype + 2^16 * textpos;
 
   c2 = Point{P_WS_1_2};
   c3 = Point{P_WS_1_3};
   c6 = Point{P_WS_1_6};
   c7 = Point{P_WS_1_7};

   View "Constraints" {
      T3(c2[0], c2[1], c2[2], font){ "A" };
      T3(c3[0], c3[1], c3[2], font){ "B" };
      T3(c6[0], c6[1], c6[2], font){ "C" };
      T3(c7[0], c7[1], c7[2], font){ "D" };
   };

/* --------------------------------------------------------------------
   Mesh
-------------------------------------------------------------------- */

   Stringers = {su_1[], su_2[], su_3[], su_4[], 
                sl_1[], sl_2[], sl_3[], sl_4[]};

   Mesh.RecombineAll = 1;

   Transfinite Line "*" = 2;
   Transfinite Line{Stringers[]} = 5;

   Transfinite Surface "*";
