# Example: Swept Wing
#
# Solid model: 
#  - Definition of the solid model
#  - Computation of first 10 normal modes
#  - Storage of solid component in binary file solid.bin
#  - Storage of translation data in binary file data.bin
#
# ---------------------------------------------------------------------

  fid  = fopen("solid.res", "wt");

  nofmod = 10;      % Number of normal modes

# Model Definition
# ----------------

# Parameters

  params

# Translation data

  data = struct("type", "solid", "subtype", "3d");

  mat = struct("type", "iso", "E", E, "ny", ny, "rho", rho);

# Ribs

  [geom, e] = mfs_beamsection("T", ribs.b, ribs.h, ribs.t);
  geom.v    = [0, 0, -1];
  geom.P    = [0, -e(1)];
  Ribs_Upper_Part = struct("type", "elements", "name", "b2",
                           "geom", geom, "mat", mat);

  Ribs_Lower_Part        = Ribs_Upper_Part;
  Ribs_Lower_Part.geom.v = [0, 0, 1];

  data.Ribs_Upper_Part = Ribs_Upper_Part;
  data.Ribs_Lower_Part = Ribs_Lower_Part;

  Ribs_Front_Part = Ribs_Upper_Part;
  Ribs_Front_Part.geom.v = [1, 0, 0];

  Ribs_End_Part = Ribs_Front_Part;
  Ribs_End_Part.geom.v = [-1, 0, 0];

  data.Ribs_Front_Part = Ribs_Front_Part;
  data.Ribs_End_Part   = Ribs_End_Part;

  geoms = struct("t", t_ribs);

  data.Ribs = struct("type", "elements", "name", "s4",
                     "geom", geoms, "mat", mat);

# Stringers

  Stringer = struct("type", "elements", "name", "b2",
                    "mat", mat);

  for n = 1 : 4

      [geom, e] = mfs_beamsection("T", stringer(n).b,
                                       stringer(n).h,
                                       stringer(n).t);
      geom.v = [0, 0, -1];
      geom.P = [0, -e(1)];
      Stringer.geom = geom;

      Upper = sprintf("Stringer_%d_Upper", n);
      Lower = sprintf("Stringer_%d_Lower", n);

      data.(Upper) = Stringer;
      Stringer.geom.v = [0, 0, 1];
      data.(Lower) = Stringer;

  endfor

# Spars

  Spar_Names = {"Front_Spar", "Main_Spar", "Back_Spar", ...
                "Rear_Spar"};

  Spar = struct("type", "elements", "name", "s4",
                "mat", mat);

  for n = 1 : 4
      Spar.geom = struct("t", t_spars(n));
      data.(Spar_Names{n}) = Spar;
  endfor

# Skin

  geoms = struct("t", t_skin);

  data.Skin = struct("type", "elements", "name", "s4",
                     "geom", geoms, "mat", mat);

# Constraints

  data.Clamped = struct("type", "constraints",
                   "name", "prescribed",
                   "dofs", 1 : 6);

# Analysis
# --------

  model = mfs_import(fid, "solid.msh", "msh", data);
  wings = mfs_new(fid, model);
  mfs_export("solid.axes", "msh", wings, "mesh", "axes");

  wings = mfs_stiff(wings);
  wings = mfs_mass(wings);
  mfs_massproperties(fid, wings);

  save -binary solid.bin wings
  save -binary data.bin data

  wings = mfs_freevib(wings, nofmod);
  mfs_print(fid, wings, "modes", "freq");
  mfs_export("modes.dsp", "msh", wings, "modes", "disp");

  fclose(fid);
