# Example: Swept Wing
#
# Splines:
#  - Definition of the aeroelastic model
#  - Transfer of aerodynamic loads to solid model
#  - Computation of displacements
#  - Storage of aeroelastic model and component in binary file 
#    aeroelastic.bin
#
# Files needed: solid.bin, aero.bin
#
# ---------------------------------------------------------------------

  fid  = fopen("splines.res", "wt");

# Aeroelastic model
# -----------------

  params

  load solid.bin
  load aero.bin

  model = struct("type", "aeroelastic", 
                 "solid", wings, "aero", winga);

# Splines

  data = struct("nbreaks", nsb);

  model.splines = struct("id", 1,       "type", "tb",
                         "lsid", 1 : 3, "data", data);

# Create component and compute splines

  wing = mfs_new(fid, model);
  wing = mfs_splines(wing);

  save -binary aeroelastic.bin model wing

# Transfer aerodynamic loads from rigid aerodynamic analysis
# to the solid component and retrieve load resultants

  wings = mfs_transfer(wing, winga, "statresp", "loads");
  [FA, MA] = mfs_getresp(winga, "statresp", "aeload");
  [FS, MS] = mfs_getresp(wings, "load", "resultant");

  nc = columns(FA);

# Compute aerodynamic coefficients

  A = mfs_getresp(winga, "mesh", "area");

  cLA = FA(3, :) / A;
  cMA = MA(1, :) / (yt * A);
  cLS = FS(3, :) / A;
  cMS = MS(1, :) / (yt * A);

  fprintf(fid, "Aerodynamic coefficients:\n\n");
  fprintf(fid, "Config.   cLA     cLS    cMxA    cMxS  \n");

  for n = 1 : nc
      fprintf(fid, "  %2.0d    %6.4f  %6.4f  %6.4f  %6.4f\n",
              n, cLA(n), cLS(n), cMA(n), cMS(n));
  endfor
  
# Compute displacements and transfer them to the aerodynamic
# component

  wings = mfs_statresp(wings);
  winga = mfs_transfer(wing, wings, "statresp", "disp");

  mfs_export("solid.lds", "msh", wings, "load", "point");
  mfs_export("solid.dsp", "msh", wings, "statresp", "disp");
  mfs_export("aero.dsp", "msh", winga, "statresp", "disp");

# Combine the models and the displacements

  mfs_merge("solid.msh", "aero.msh", "wing.msh", "msh");
  mfs_merge("solid.dsp", "aero.dsp", "wing.dsp", "msh");

  fclose(fid);
