# Example: Free vibrations of an arc
#          2-dimensional model with membrane elements
#
# -------------------------------------------------------------------

# Data (N, mm)

  E   =  210000; % Young's modulus
  ny  =     0.3; % Poisson ratio
  rho = 7.85E-9; % Mass density
  t   =       2;

  nmodes   = 10;
  elnam    = {"2 = t3", "3 = q4", "9 = t6", "10 = q9", "16 = q8"};

# Type of mass matrix

  masstype = {"consistent", "lumped"};
  [mtp, ok] = listdlg("Name", "Select mass type",
                      "ListString", masstype,
                      "SelectionMode", "single",
                      "ListSize", [200, 50]);  
  if (! ok) exit; endif

  fid = fopen("arc.res", "wt");

  fprintf(fid, "Mass type = %s\n\n", masstype{mtp});

# Translation data

  model = struct("type", "solid", "subtype", "2d");

  geom = struct("t", t);
  mat  = struct("type", "iso", "E", E, "ny", ny, "rho", rho);

  model.Elements = struct("type", "elements",
                          "name", {elnam},
                          "geom", geom,
                          "mat",  mat);

  model.Constraints = struct("type", "constraints",
                             "name", "prescribed",
                             "dofs", [1, 2]);

# Import the msh-file

  input = mfs_import(fid, "arc.msh", "msh", model);

# Initialize component

  arc = mfs_new(fid, input);

# Stiffness and mass matrices

  arc = mfs_stiff(arc);
  arc = mfs_mass(arc, masstype{mtp});
  mp = mfs_massproperties(fid, arc);

# Free vibrations

  arc = mfs_freevib(arc, nmodes);

# Output

  mfs_print(fid, arc, "modes", "freq");
  mfs_export("arc.pos", "msh", arc, "modes", "disp");

# Modal effective masses

  mfs_meffmass(fid, arc, mp.cm);

  fclose(fid);
