# Example: Box Beam
#          Transient response due to base motion: restart
#
# -------------------------------------------------------------------

  colors = [1, 0, 0; 0, 1, 0; 0, 0, 1; 1, 0, 1];
  set(0, "defaultaxescolororder", colors);

# Output device depending on operating system

  if (isunix)
     DEVICE = "svg";
  else
     DEVICE = "png";
  endif
  FORMAT = ["-d", DEVICE];
  EXT    = [".", DEVICE];

# Define restart parameters

  t1 = 0.05;   % Time at which to restart
  dt = 5e-4;   % Time step for restart
  T  =  0.1;   % Simulation time

  Te = T + t1; % Total time

# Load results

  if (! isfile("tresp.bin"))
     error("File tresp.m must be executed before\n");
  endif

  load tresp.bin

# Get restart data

  t = mfs_getresp(beam, "transresp", "time");
  stepno = lookup(t, t1);

  fixed = struct("lc", 1, "func", "fixed");

# Restart transient response analysis

  beam = mfs_transresp(beam, [dt, T], "transresp", [1, stepno],
                      "load", fixed, "rcase", 2);

# Plot results at point B

  ridB = {"Point_B", 3};

  tr = t1 + mfs_getresp(beam, "transresp", "time", 2); 

  u  = mfs_getresp(beam, "transresp", "disp", ridB, 1);
  ur = mfs_getresp(beam, "transresp", "disp", ridB, 2);
  v  = mfs_getresp(beam, "transresp", "velo", ridB, 1) / 1000;
  vr = mfs_getresp(beam, "transresp", "velo", ridB, 2) / 1000;
  a  = mfs_getresp(beam, "transresp", "acce", ridB, 1) / 9810;
  ar = mfs_getresp(beam, "transresp", "acce", ridB, 2) / 9810;

  figure(1, "position", [100, 200, 1000, 800],
            "paperposition", [0, 0, 15, 15]);
  subplot(3, 1, 1)
     plot(t, u, tr, ur);
     legend("initial", "restart", "location", "southeast");
     axis("labely");
     grid; xlim([0, Te]);
     ylabel('u [mm]');
  subplot(3, 1, 2)
     plot(t, v, tr, vr);
     axis("labely");
     grid; xlim([0, Te]);
     ylabel('v [m/s]');
  subplot(3, 1, 3)
     plot(t, a, tr, ar);
     grid; xlim([0, Te]);
     ylabel('a [g]'); xlabel('t [s]');
  print(["restart", EXT], FORMAT);
