# Example: Plate
#          Analysis of Excitation Functions
#
# --------------------------------------------------------

  pkg load signal

# Output device depending on operating system

  if (isunix)
     DEVICE = "svg";
  else
     DEVICE = "png";
  endif
  FORMAT = ["-d", DEVICE];
  EXT    = [".", DEVICE];

# Pulse Function
# --------------

# Data

  T     =   0.01;   % Duration of pulse
  Ts    = 10 * T;   % Length of time signal
  dt    = T / 20;   % Time step
  delay =   0.01;   % Delay
  fc    =    450;   % Cut-off frequency

# Function values

  t = 0 : dt : Ts;
  N = length(t);
  y = pulse(t, [T, delay]);

# Fourier transform

  fs = 1 / dt; fa = 0.5 * fs;
  Y  = fft(y) * dt;
  f  = (0 : N - 1) / Ts;

# Plot function and its Fourier transform

  figure(1, "position", [100, 100, 750, 750],
            "paperposition", [0, 0, 15, 12]);
  subplot(2, 1, 1);
     plot(t, y, "color", "green");
     grid;
     xlabel('t [s]'); ylabel('\phi(t)');
  subplot(2, 1, 2);
     plot(f, 100 * abs(Y), "color", "red");
     grid;
     xlim([0, fa]);
     xlabel('f [Hz]'); ylabel('|100 \Phi(f)| [1/Hz]');
  print(["pulse", EXT], FORMAT);

# Test of cut-off frequency

  yc = resample(y, fc, fa);
  tc = (0 : length(yc) - 1) * dt * fa / fc;

  figure(2, "position", [500, 100, 750, 500],
            "paperposition", [0, 0, 15, 10]);
  plot(t, y, "color", "green", tc, yc, "color", "red");
  legend("orig.", "filtered");
  grid;
  xlim([0, 2 * T + delay]);
  xlabel('t [s]'); ylabel('\phi(t)');
  print(["cutoff", EXT], FORMAT);

# Decay Function
# --------------

  clear t; clear y; clear Y; clear f;
  clear yc; clear tc;

  T     =   0.01;   % Duration of pulse
  Ts    = 10 * T;   % Length of time signal
  dt    = T / 20;   % Time step
  fc    =    450;   % Cut-off frequency

# Function values

  t = 0 : dt : Ts;
  N = length(t);
  y = decay(t, T);

# Fourier transform

  fs = 1 / dt; fa = 0.5 * fs;
  Y  = fft(y) * dt;
  f  = (0 : N - 1) / Ts;

# Plot function and its Fourier transform

  figure(3, "position", [100, 200, 750, 750],
            "paperposition", [0, 0, 15, 12]);
  subplot(2, 1, 1);
     plot(t, y, "color", "green");
     grid;
     xlabel('t [s]'); ylabel('\phi(t)');
  subplot(2, 1, 2);
     plot(f, 100 * abs(Y), "color", "red");
     grid;
     xlim([0, fa]);
     xlabel('f [Hz]'); ylabel('|100 \Phi(f)| [1/Hz]');
  print(["decay", EXT], FORMAT);

# Test of cut-off frequency

  yc = resample(y, fc, fa);
  tc = (0 : length(yc) - 1) * dt * fa / fc;

  figure(4, "position", [500, 200, 750, 500],
            "paperposition", [0, 0, 15, 10]);
  plot(t, y, "color", "green", tc, yc, "color", "red");
  legend("orig.", "filtered");
  grid;
  xlim([0, 2 * T + delay]);
  xlabel('t [s]'); ylabel('\phi(t)');
  print(["cutoff_decay", EXT], FORMAT);
