# Example: Plate
#          Direct transient response analysis
#          Initial deformation
#
# -------------------------------------------------------------------

  colors = [1, 0, 0; 0, 1, 0; 0, 0, 1];
  set(0, "defaultaxescolororder", colors);

  if (isunix)
     DEVICE = "svg";
  else
     DEVICE = "png";
  endif
  FORMAT = ["-d", DEVICE];
  EXT    = [".", DEVICE];

# Define excitation parameters

  T      =  2e-3;  % Duration of decay

# Define simulation paramters

  dt     =  1e-4;  % Time step
  Ts     =  0.15;  % Simulation time

# Define excitation

  loads = struct("lc", 1, "func", "decay", "params", T);

# Get results from normal modes analysis

  if (! isfile("modes.bin"))
     error("File modes.m must be executed before\n");
  end

  load modes.bin

# Compute initial deformation

  plate = mfs_statresp(plate);

# Perform transient response analysis

  plate = mfs_transresp(plate, [dt, Ts], "load", loads,
                        "statresp", 1, "method", "direct");

# Plot results

  rid = {"Resp_A", 3; "Resp_B", 3; "Resp_C", 3};

  t = mfs_getresp(plate, "transresp", "time");
  w = mfs_getresp(plate, "transresp", "disp", rid);
  v = mfs_getresp(plate, "transresp", "velo", rid) / 1000;
  a = mfs_getresp(plate, "transresp", "acce", rid) / 9810;

  figure(1, "position", [100, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, w);
  legend("Pnt. A", "Pnt. B", "Pnt. C");
  grid;
  xlim([0, Ts]);
  xlabel('t [s]'); ylabel('w [mm]');

  print(["decay_direct_w", EXT], FORMAT);

  figure(2, "position", [300, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, v);
  legend("Pnt. A", "Pnt. B", "Pnt. C");
  grid;
  xlim([0, Ts]);
  xlabel('t [s]'); ylabel('v [m/s]');

  print(["decay_direct_v", EXT], FORMAT);

  figure(3, "position", [500, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, a);
  legend("Pnt. A", "Pnt. B", "Pnt. C",
         "location", "southeast");
  grid;
  xlim([0, Ts]);
  xlabel('t [s]'); ylabel('a [g]');

  print(["decay_direct_a", EXT], FORMAT);

# Save results

  dlmwrite("decay_direct.csv", [t; w; v; a], "precision", 5);
