# Example: Plate
#          Direct transient response analysis
#          Force excitation
#
# ----------------------------------------------------------------

  colors = [1, 0, 0; 0, 1, 0; 0, 0, 1];
  set(0, "defaultaxescolororder", colors);

# Output device depending on operating system

  if (isunix)
     DEVICE = "svg";
  else
     DEVICE = "png";
  endif
  FORMAT = ["-d", DEVICE];
  EXT    = [".", DEVICE];

# Define excitation parameters

  T     =          0.01;  % Duration of pulse
  delay = [0, 4, 2] * T;  % Delays

# Define simulation paramters

  dt     =  1e-4;  % Time step
  Ts     =  0.15;  % Simulation time

# Define excitation

  for l = 1 : 3
      loads(l) = struct("lc", l, "func", "pulse",
                        "params", [T, delay(l)]);
  end

# Get component with matrices

  if (! isfile("modes.bin"))
     error("File modes.m must be executed before\n");
  end

  load modes.bin

# Perform transient response analysis

  plate = mfs_transresp(plate, [dt, Ts], "load", loads,
                        "method", "direct");

# Plot results

  rid = {"Resp_A", 3; "Resp_B", 3; "Resp_C", 3};

  t = mfs_getresp(plate, "transresp", "time");
  w = mfs_getresp(plate, "transresp", "disp", rid);
  v = mfs_getresp(plate, "transresp", "velo", rid) / 1000;
  a = mfs_getresp(plate, "transresp", "acce", rid) / 9810;

  figure(1, "position", [100, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, w);
  legend("Pnt. A", "Pnt. B", "Pnt. C", "location", "southeast");
  legend("autoupdate", "off");
  grid;
  xlim([0, Ts]);
  line([delay(2), delay(2)], ylim(), "color", "black");
  line([delay(3), delay(3)], ylim(), "color", "black");
  xlabel('t [s]'); ylabel('w [mm]');

  print(["impact_direct_w", EXT], FORMAT);

  figure(2, "position", [300, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, v);
  legend("Pnt. A", "Pnt. B", "Pnt. C");
  legend("autoupdate", "off");
  grid;
  xlim([0, Ts]);
  line([delay(2), delay(2)], ylim(), "color", "black");
  line([delay(3), delay(3)], ylim(), "color", "black");
  xlabel('t [s]'); ylabel('v [m/s]');

  print(["impact_direct_v", EXT], FORMAT);

  figure(3, "position", [500, 200, 1000, 750],
            "paperposition", [0, 0, 15, 12]);
  plot(t, a);
  legend("Pnt. A", "Pnt. B", "Pnt. C");
  legend("autoupdate", "off");
  grid;
  xlim([0, Ts]);
  line([delay(2), delay(2)], ylim(), "color", "black");
  line([delay(3), delay(3)], ylim(), "color", "black");
  xlabel('t [s]'); ylabel('a [g]');

  print(["impact_direct_a", EXT], FORMAT);

# Save results

  dlmwrite("impact_direct.csv", [t; w; v; a], "precision", 5);
