function [cmp, out, rc] = mfs_backx(cmp, class, item, args)

# usage: [cmp, out, rc] = mfs_backx(cmp, class, item, args)
#
# Input  cmp       Structure with component data
#        class     Response class
#        item      Response item
#        args{:}   Additional arguments depending on class and item
#
# Output cmp       Structure with component data
#        out       Additional output 
#        rc        Return code: 0 = no errors
#                               1 = errors 
#
# The functions performs the back transformation from modal to
# physical coordinates for aerodynamic components.
#
# class = freqresp
#
#   item   Description      Additional arguments
#   -------------------------------------------------------------------
#   disp   Displacements    args{1}   Load case number
#                           args{2}   List of excitation frequencies
#
# ------------------------------------------------------------------------

# Copyright (c) 2020 by Johannes Wandinger

# Check arguments

  if (nargin != 4 || nargout != 3)
     print_usage();
  end

# Initialize

  out = [];
  rc  = 0;

# Branch according to response class

  switch class

# Frequency response

  case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*E* mfs_back: no frequency response results found\n");
        rc = 1; return;
     end

     if (isempty(ldc = args{1}))
        printf("*E* mfs_back: loadcase not defined\n");
        rc = 1; return;
     end
     if (ldc < 0 || ldc > cmp.freqresp.nfa)
        printf("*E* mfs_back: loadcase %d does not exist\n", ldc);
        rc = 1; return;
     end
     if (isempty(resp = cmp.freqresp.resp{ldc}))
        printf("*E* mfs_back: no data found for loadcase %d\n", ldc);
        rc = 1; return;
     end

     if (isempty(f = args{2}))
        printf("*E* mfs_back: no frequencies defined\n");
        rc = 1; return;
     end
     if (f(1) < resp.freq(1) || f(end) > resp.freq(end))
        printf("*E* mfs_back: frequencies outside range ");
        printf("%7.2f < f < %7.2f\n", resp.freq(1), resp.freq(end));
        rc = 1; return;
     end

     if (resp.method == 1)
        printf("*E* mfs_back: results of direct frequency response");
        printf("analyis found\n");
        printf("              no backtransformation needed\n");
        rc = 1; return;
     end

     f     = sort(f);
     nback = length(f);

     ftest = [resp.freq, 10 * resp.freq(end)];
     ixl   = lookup(ftest, f);
     ixu   = ixl + 1;
     df    = [f - resp.freq(ixl); resp.freq(ixu) - f];
     [m, im] = min(df);
     ix  = [ixl; ixu];
     ixb = zeros(1, nback);
     for k = 1 : nback
         ixb(k) = ix(im(k), k);
     end
     freqback = ftest(ixb);
     out      = freqback;

     cmp.freqresp.resp{ldc}.nback    = nback;
     cmp.freqresp.resp{ldc}.freqback = freqback;

#    Process item

     switch item

     case "disp"

        if (cmp.freqresp.resp{ldc}.type == 11)     % Aeroelasticity
           if (cmp.freqresp.resp{ldc}.method == 4)
              U = cmp.freqresp.resp{ldc}.Ue(:, ixb);
              rigmod = 1 : cmp.modes.rigmod;
              U += cmp.modes.disp(:, rigmod) ...
                 * cmp.freqresp.resp{ldc}.Q(rigmod, ixb);
           else
              U = cmp.modes.disp * cmp.freqresp.resp{ldc}.Q(:, ixb);
           end
        end

        cmp.freqresp.resp{ldc}.U = U;

     otherwise

        printf("*E* mfs_back: unknown response item \"%s\"\n", item);
        rc = 1;

     end    % item

  otherwise

     printf("*E* mfs_back: unknown response class \"%s\"\n", class);
     rc = 1;

  end      % class

end
