function [pids, rc] = mfs_findpanelcol(ls, lsids, ycolno)

# usage: [pids, rc] = mfs_findpanelcol(ls, lsids, ycolno)
#
# Input  ls         Structure with lifting surface data
#        lsids(:)   Array with lifting surface identifiers
#        ycolno     Number of panel column
#
# Output pids(:)    Array with panel indices
#        rc         Return code: 0 = no errors
#
# The function returns the indices of the panels in one panel column.
# Panel columns run in x-direction.
#
# ---------------------------------------------------------------------

# Check arguments

  if (nargin != 3 || nargout !=2)
     print_usage();
  end

# Initialize

  pids = [];
  rc = 1;

# Check if lifting surfaces do exist

  ixs = lookup(ls.ids, lsids, "m");

  ixbad = find(ixs == 0);
  nbad  = length(ixbad);
  if (nbad)
     for ixb = ixbad
         printf("*E* mfs_xydata: lifing surface %d does not exist\n",
                lsids(ixb));
     end
     return;
  end

  rc = 0;

# Check compatibility of lifting surfaces

  nofls = length(ixs);
  ny = ls.surfs(ixs(1)).ny;
  for l = 2 : nofls
      if (ny != ls.surfs(ixs(l)).ny)
         printf("*E* mfs_xydata: lifting surfaces are not compatible\n");
         printf("    Lifting surface %5.0d: ny = %5.0d\n", 
                lsids(1), ny);
         printf("    Lifting surface %5.0d: ny = %5.0d\n", 
                lsids(l), ls.surfs(ixs(l)).ny);
         rc = 1;
      end
  end
  
  if (rc) return; end

# Check column number

  if (ycolno > ny || ycolno < 1)
     rc = 1;
     printf("*E* mfs_xydata: illegal column number:");
     printf(" ycolno = %d, ny = %d\n", ycolno, ny);
     return;
  end

# Process lifting surfaces

  for l = 1 : nofls
      ix  = ixs(l);
      nx  = ls.surfs(ix).nx;
      p1  = ls.surfs(ix).p1 + (ycolno - 1) * nx;
      p2  = p1 + nx - 1;
      pids = [pids, p1 : p2];
  end

end
