function [cmp, rc] = mfs_freqrespx(cmp, kred, opts)

# usage: [cmp, rc] = mfs_freqrespx(cmp, kred, opts)
#
# Input  cmp     Structure with component data
#        kred(:) List of reduced frequencies
#        opts    Structure with wake options
#
# Output cmp     Structure with component data
#        rc      Return code: 0 = no errors
#                             1 = errors
#
# The function computes the the frequency response of an aerodynamic
# component.
#
# Methods supported: vortex-lattice method
#
# --------------------------------------------------------------------

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  endif

# Initialize

  rc = 0;
  np = cmp.panels.nofpan;
  nk = length(kred);

# Check if reference chord is defined

  if (! cmp.cref)
     printf("*E** mfs_freqresp: Reference chord lenght undefined\n");
     rc = 1;
  endif

# Check if motion is defined

  if (! isfield(cmp, "motion"))
     printf("*E* mfs_freqresp: Motion is not defined\n");
     rc = 1;
  endif

  if (rc) return; endif

# Make sure reduced frequencies are ascending

  kred = sort(kred);

# Initialize result

  G = complex(zeros(np, nk));

# Get information on symmetry

  symy = isfield(cmp, "symy");
  if (symy)
     ys = cmp.symy;
  else
     ys = 0;
  endif

# Build matrices that do not depend on reduced frequency and
# initialize the wake grid

  [C, S, wake] = mfs_vlmhpre(cmp.ls, cmp.panels, cmp.pancols, symy, ys,
                             cmp.cref, opts);
  [CL, CU] = lu(C); clear C;
  unit     = eye(rows(S));

# Loop over reduced frequencies

  for n = 1 : nk

      F = mfs_vlmhwake(wake, kred(n), cmp.panels.nofpan, 
                       cmp.pancols.npcol, opts.ktol);
      W = mfs_vlmdwh(cmp.panels, cmp.controls, kred(n), cmp.motion,
                      cmp.cref);

      GF = CU \ (CL \ [W, F]);
      G(:, n) = GF(:, 1);
      F       = GF(:, 2 : end);
      clear wf;

      R        = unit - S * F;
      dG       = R \ (S * G(:, n));
      G(:, n) += F * dG;

  endfor

# Store results

  resp = struct("method", 1, "type", 1, "nfreq", nk, "kred", kred,
                "G", G);
  cmp.freqresp.nfa     = 1;
  cmp.freqresp.resp{1} = resp;

endfunction
