function [camber, rc] = mfs_naca5(digits)

# usage: camber = mfs_naca5(digits)
#
# Input  digits   Digits 2 and 3 of airfoil definition
#
# Output camber   Piecewise polynomial defining camber
#        rc       Return code: 0 = no errors
#
# --------------------------------------------------------------------

# NACA 5 data (cf. http://airfoiltools.com/airfoil/naca4digit)

  NACA5.r  = [0.0580, 0.1260, 0.2025, 0.2900, 0.3910;
                   0, 0.1300, 0.2170, 0.3180, 0.4410];
  NACA5.k1 = [361.400, 51.640, 15.957, 6.643, 3.230;
                    0, 51.990, 15.793, 6.520, 3.191];
  NACA5.k2 = [ 0, 0.000764, 0.00677, 0.0303, 0.1355];

# Check arguments

  if (nargin != 1 || nargout != 2)
     print_usage();
  end

# Initialize

  camber = struct();
  rc     = 0;

# Check input data

  P = floor(digits / 10);
  Q = mod(digits, 10);

  if (P < 1 || P > 5 || Q < 0 || Q > 1)
     printf("*E* mfs_airfoil: NACA series 5: digits %2.0d not supported\n",
            digits);
     rc = 1; return;
  end

  if (Q == 1 && P == 1)
     printf("*E* mfs_airfoil: NACA series 5: digits %2.0d not supported\n",
            digits);
     rc = 1; return;
  end

# Define camber

  Q1 = Q + 1;
  r  = NACA5.r(Q1, P);
  k1 = NACA5.k1(Q1, P);
  r3 = r^3;

  if (Q == 0)
     c = [1, -3 * r,  r^2 * (3 - r), 0;
          0, 0, -r3, r3];
  else
     k2 = NACA5.k2(P);
     d  = k2 * (1 - r)^3 + r^3;
     c = [1, -3 * r, 3 * r^2 - d, 0;
          k2, -3 * k2 * r, 3 * k2 * r^2 - d, (1 - k2) * r^3];
  end

  b = [0, r, 1];
  c = c * k1 / 6;

  c(2, :) = polyaffine(c(2, :), [-r, 1]);

  camber = mkpp(b, c);

end



