function [motion, rc] = mfs_new_motion(msg, mot, controls)

# usage: [motion, rc] = mfs_new_motion(msg, mot, controls)
#
# Input  msg        File handle of message file
#        mot        Structure with definition of motion
#        controls   Structure with control surfacces
#                   (empty if there aren't any)
#
# Output motion     Structure with definition of motion
#        rc         Return code: 0 = no errors
#                                1 = errors
#
# The function builds the structure motion from the input data.
#
# The following fields are defined:
#
# motion.refpnt(3)              Coordinates of reference point
#       .heave                  Heave amplitude
#       .pitch                  Pitch amplitude
#       .<cntnam>               Amplitude of control surface angle
#
# --------------------------------------------------------------------

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Initialize

  rc     = 0;
  motion = struct();

  if ((ncntrl = controls.ncntrl))
     fldnam = fieldnames(controls);
     cntnam = fldnam(2 : end);
  else
     cntnam = [];
  end

  motion.refpnt = [0, 0, 0];

# Process the definitions

  for [val, key] = mot 
      switch key
      case "refpnt"
         if (length(val) != 3)
            fprintf(msg, "*E* Bad definition of motion.refpnt\n");
            rc = 1;
         else
            motion.refpnt = val;
         end
      case "heave"
         if (length(val) != 1)
            fprintf(msg, "*E* Bad definition of motion.heave\n");
            rc = 1;
         else
            motion.heave = val;
         end
      case "pitch"
         if (length(val) != 1)
            fprintf(msg, "*E* Bad definition of motion.pitch\n");
            rc = 1;
         else
            motion.pitch = val;
         end
      case "roll"
         if (length(val) != 1)
            fprintf(msg, "*E* Bad definition of motion.roll\n");
            rc = 1;
         else
            motion.roll = val;
         end
      case cntnam
         if (length(val) != 1)
            fprintf(msg, "*E* Bad definition of motion.%s\n",
                    key);
            rc = 1;
         else
            motion.(key) = val;
         end
      otherwise
         rc = 1;
         fprintf(msg, "*E* motion: illegal field \"%s\"\n", key);
      end
  end

end
