function [points, rc] = mfs_new_points(msg, input_points)

# usage: [points, rc] = mfs_new_points(msg, input_points)
#
# Input  msg           File handle of message file
#        input_nodes   Structure array with input data of the points
#
# Output points        Structure with point data
#        rc            Return code: 0 means no erros
#                                   1 means errors
#
# The function builds a structure with all point data from the
# input data.
#
# Input data:
#
#    input_points(:).id         Point identifier
#    input_points(:).coor(3)    Point coordinates
#
# Output data:
#
#    points.nofpnt            Number of points
#    points.ids(:)            Point identifiers in ascending order
#    points.coor(:, 3)        Point coordinates 
#
# ---------------------------------------------------------------------

  point_fields = {"id", "coor"};

# Check arguments

  if (nargin != 2 || nargout != 2)
     print_usage();
  end

# Initialize

  rc     = 0;
  points = struct();

# Check input data

  if ((rc = mfs_checkfields1(msg, input_points, point_fields, 
                             "Points")))
     return;
  end

# Sort point identifiers

  nofpnt = length(input_points);
  pntids = zeros(1, nofpnt);

  for n = 1 : nofpnt
      if (input_points(n).id)
         pntids(n)  = input_points(n).id;
      else
         rc = 1;
         fprintf(msg, "*E* Identifier of point number %5d undefined\n", n)
      end
  end

  if (rc)
     return;
  end

  [pntids, index] = sort(pntids);

# Check on duplicate point identifiers

  for n = 1 : nofpnt - 1
      if (pntids(n) == pntids(n + 1))
         rc = 1;
         fprintf(msg, "*E* Duplicate point identifier %5d found\n",
                 pntids(n));
      end
  end

# Get coordinates

  coor = zeros(nofpnt, 3);
  for n = 1 : nofpnt
      ix = index(n);
      if (isvector(input_points(ix).coor))
         if (length(input_points(ix).coor) == 3)
            coor(n, :) = input_points(ix).coor;
         else
            rc = 1;
            fprintf(msg, "*E* Wrong definition of coordinates of point %5d\n",
                    input_points(ix).id);
         end
      else
         rc = 1;
         fprintf(msg, "*E* Coordinates of point %5d undefined\n",
                input_points(ix).id);
      end
  end

  if (rc)
     return;
  end

# Build the point structure

  points = struct("nofpnt", nofpnt, "ids", pntids, "coor", coor);

end
