function [rbmp, rc] = mfs_new_rbmp(msg, mass)

# usage: [rbmp, rc] = mfs_new_rbmp(msg, mass)
#
# Input  mass     Structure with rigid body mass properties
# 
# Output rbmp     Structure with rigid body mass properties
#        rc       Return code: 0 means no erros
#                              1 means errors
#
# The function checks fields m, JS and cm and transfers them to 
# structure rbmp.
#
# --------------------------------------------------------------------

  req_fields = {"m", "JS", "cm"};

# Check arguments

  if (nargin != 2 || nargout != 2)
     print_usage();
  end

# Initialize

  rbmp = struct();

# Check input data

  if ((rc = mfs_checkfields1(msg, mass, req_fields, "Mass")))
     return;
  end

# Mass must be positive

  if (mass.m > 0)
     rbmp.m = mass.m;
  else
     rc = 1;
     fprintf(msg, "*E* Rigid body mass not positive\n");
  end

# Inertia tensor must be positive definite

  [mr, nc] = size(mass.JS);
  if (mr != 3 || nc != 3)
     rc = 1;
     fprintf(msg, "*E* Inertia tensor not a 3x3 matrix\n");
  elseif (det(mass.JS) > 0)
     rbmp.JS = mass.JS;
  else
     rc = 1;
     fprintf(msg, "*E* Inertia tensor not positive definite\n");
  end

# Coordinates of center of mass

  if ((lcm = length(mass.cm)) == 3)
     rbmp.cm = mass.cm;
  else
     rc = 1;
     fprintf(msg, "*E* Bad definition of center of mass: ");
     fprintf(msg, "number of coordinates is %d\n", lcm);
  end

end
