function [cmp, rc] = mfs_new_vlm(msg, model, options)

# usage: [cmp, rc] = mfs_new_vlm(msg, model, options)
#
# Input  msg     File handle of message file
#        model   Structure with model data
#        options Structure with options
#
# Output cmp     Structure with component data
#                (0 if errors occurred)
#        rc      Return code: 0 = no errors
#                             1 = errors
#
# The function initializes a new component of type aero and subtype vlm
# (Vortex lattice method). The following fields of are defined:
#
#  type       Component type
#  subtype    Component subtype
#  points     Structure with point data
#  ls         Structure with lifting surface data
#  panels     Structure with panel data
#  pancols    Structure with panel column data
#  controls   Structure with data of control surfaces
#  symy       Field indicating symmetry with respect to xz-plane
#  config     Structure with configuration data
#  motion     Structure with definition of motion
#  rbmp       Structure with rigid body mass properties
#  cref       Reference chord length
#
# --------------------------------------------------------------------

  legal_fields = {"type", "subtype", "points", "ls", "wake", ...
                  "symy", "cref", "controls", "config", "motion", ...
                  "mass"};

  cmp = struct();

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Check fields

  if ((rc = mfs_checkfields4(msg, model, legal_fields,
                             "Model definition")))
     return;
  end

# Initialize

  fprintf(msg, "\n  Model Type = %s, Model Subtype = %s\n\n",
          model.type, model.subtype);

# Points

  if (isfield(model, "points"))
     [points, rc] = mfs_new_points(msg, model.points);
  else
     rc = 1;
     fprintf(msg, "*E* Definition of points missing\n");
  end

  if (rc) return; end

# Lifting surfaces

  if (isfield(model, "ls"))
     [ls, panels, nodes, rc] = mfs_new_ls(msg, points, model.ls);
  else
     rc = 1;
     fprintf(msg, "*E* Definition of lifting surfaces missing\n");
  end

  if (rc) return; end

# Panel columns

  [pancols, ls, rc] = mfs_new_pancols(msg, ls, panels, nodes, 
                                      options.rtolp);

# Controls

  if ((iscntl = isfield(model, "controls")))
     [controls, rc] = mfs_new_controls(msg, ls, model.controls);
     if (rc) return; end
  else
     controls = struct("ncntrl", 0);
  end

# Configurations

  if (isfield(model, "config"))
     [config, rc] = mfs_new_config(msg, model.config, ls, controls);
  else
     config.nconf = 0;
  end

# Motion

  if (isfield(model, "motion"))
     defmot = 1;
     [motion, rcx] = mfs_new_motion(msg, model.motion, controls);
     rc = rc || rcx;
  else
     defmot = 0;
  end

# Symmetry

  symy = isfield(model, "symy");
  if (symy)
     y0 = model.symy;
     if (y0 > nodes.mincor(2) && y0 < nodes.maxcor(2))
        rc = 1;
        fprintf(msg, "*E* Symmetry plane cuts the model\n");
     end
  end

# Reference chord

  if(isfield(model, "cref"));
     if ((cref = model.cref) <= 0)
        rc = 1;
        fprintf(msg, "*E* Reference chord is not positive\n");
     end
  else
     cref = 0;
  end

# Rigid body mass properties

  if ((ismass = isfield(model, "mass")))
     [rbmp, rcx] = mfs_new_rbmp(msg, model.mass);
     rc = rc || rcx;
  end

  if (rc) return; end

# Build the component

  cmp = struct("type", "aero", "subtype", "vlm", "cref", cref,
               "nodes", nodes, "ls", ls, "panels", panels, 
               "pancols", pancols, "controls", controls, 
               "config", config);
  if (defmot)
     cmp.motion = motion;
  end
  if (symy)
     cmp.symy = y0;
  end
  if (ismass)
     cmp.rbmp = rbmp;
  end

# Print messages

  fprintf(msg, "  Number of nodes  = %5d,", cmp.nodes.nofnod);
  fprintf(msg, "  Number of panels = %5d\n", cmp.panels.nofpan);
  fprintf(msg, "  Number of lifting surfaces = %12d\n", cmp.ls.nofls);
  if (iscntl)
     fprintf(msg, "  Number of control surfaces = %12d\n",
             cmp.controls.ncntrl);
  end
  fprintf(msg, "  Number of configurations   = %12d\n",
          cmp.config.nconf);
  if (isfield(model, "wake"))
     fprintf(msg, "  Number of wake groups      = %12d\n", 
             length(model.wake));
  end
  fprintf(msg, "  Reference chord length     = %12.5e\n", cref);

  if (symy)
     fprintf(msg, "  Symmetry plane: y          = %12.5e\n", y0);
  end

end
