function mfs_printx(fid, cmp, class, items)

# usage: mfs_printx(fid, cmp, class, items)
#
# Input  fid       File handle 
#        cmp       Structure with component
#        class     Output data class
#        items{}   List of items to be printed
#
# The function prints the results of an aerodynamic component.
#
# The items supported depend on the class:
#
#  class = statresp
#        disp      Displacements
#        vortex    Vortex strength
#        pressure  Panel pressure
#
#  class = diverg
#        qdyn      Dynamic pressure
#        disp      Displacements
#
#  class = trim
#        params    Trim parameters
#        disp      Displacements
#        vortex    Vortex strength
#        pressure  Panel pressure
#
#  class = freqresp
#        vortex    Vortex strength
#        pressure  Panel pressure
#
# ---------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger

  headd = {"ux", "uy", "uz"};

# Check arguments

  if (nargin < 4)
     print_usage();
  end

# Process parameters

  switch class

  case "statresp"

     if (! isfield(cmp, "statresp"));
        printf("*W* mfs_print: there are no items of class \"statresp\"\n");
       return;
     end      

     for k = 1 : length(items)
         itm = getitem(items{k}, cmp);
         switch itm
         case "disp"
            if (isfield(cmp.statresp, "disp"))
               mfs_prt_disp(fid, cmp.nodes, 3, cmp.statresp.disp, 
                            "Displacements of configuration",
                            headd);
            else
               printf("*W* mfs_print: no displacements found\n");
            end
         case "vortex"
            mfs_prt_scalar(fid, 1 : cmp.panels.nofpan, cmp.statresp.G,
                           "Vortex strengths");
         case "pressure"
            if (isfield(cmp.statresp, "panres"))
               mfs_prt_scalar(fid, 1 : cmp.panels.nofpan, 
                              cmp.statresp.panres.p, "Panel pressure");
            else
               printf("*W* mfs_print: no pressure found\n");
            end
         otherwise
            error("*W* mfs_print: unknown item ""%s""\n", items{k})
         end
     end     % Item


  case "diverg"

     if (! isfield(cmp, "diverg"))
        printf("*W* mfs_print: there are no items of class \"diverg\"\n");
        return;
     end

     for k = 1 : length(items)
         itm = getitem(items{k}, cmp);
         switch itm
         case "qdyn"
            mfs_prt_qdyn(fid, cmp);
         case "disp"
            mfs_prt_divmodes(fid, cmp, 3, headd);
         otherwise
            error("*W* mfs_print: unknown item ""%s""\n", items{k})
         end
     end     % Item

  case "trim"

     if (! isfield(cmp, "trim"));
        printf("*W* mfs_print: there are no items of class \"trim\"\n");
       return;
     end      

     for k = 1 : length(items)
         itm = getitem(items{k}, cmp);
         switch itm
         case "disp"
            if (isfield(cmp.trim, "disp"))
               mfs_prt_disp(fid, cmp.nodes, 3, cmp.trim.disp, 
                            "Displacements of configuration",
                            headd);
            else
               printf("*W* mfsp_print: no displacements found\n");
            end
         case "params"
            if (isfield(cmp.trim, "qr"))
               mfs_prt_trimparam(fid, cmp.trim.type, cmp.trim.tpnames, 
                                 cmp.trim.tpval, cmp.trim.isangle,
                                 cmp.config.qdyn, cmp.config.names, 
                                 cmp.trim.qr);
            else
               mfs_prt_trimparam(fid, cmp.trim.type, cmp.trim.tpnames, 
                                 cmp.trim.tpval, cmp.trim.isangle,
                                 cmp.config.qdyn, cmp.config.names);
            end
         case "vortex"
            mfs_prt_scalar(fid, 1 : cmp.panels.nofpan, cmp.trim.G,
                           "Vortex strengths");
         case "pressure"
            if (isfield(cmp.trim, "panres"))
               mfs_prt_scalar(fid, 1 : cmp.panels.nofpan, 
                              cmp.trim.panres.p, "Panel pressure");
            else
               printf("*W* mfs_print: no pressure found\n");
            end
         otherwise
            error("*W* mfs_print: unknown item ""%s""\n", items{k})
         end
     end     % Item

  case "freqresp"

     if (! isfield(cmp, "freqresp"));
        printf("*W* mfs_print: there are no items of class \"freqresp\"\n");
       return;
     end      

     for l = 1 : cmp.freqresp.nfa

         resp = cmp.freqresp.resp{l};

         for k = 1 : length(items)
             itm = getitem(items{k}, cmp);
             switch itm
             case "vortex"
                mfs_prt_complex(fid, 1 : cmp.panels.nofpan, resp.G,
                                resp.kred, "Vortex strengths");
             case "pressure"
                if (isfield(resp, "panres"))
                   mfs_prt_complex(fid, 1 : cmp.panels.nofpan, 
                                  resp.panres.p, resp.kred, 
                                  "Panel pressure");
                else
                   printf("*W* mfs_print: no pressure found for load case %2.0d\n",
                          l);
                end
             otherwise
                error("*W* mfs_print: unknown item ""%s""\n", items{k})
             end
         end     % Item

     end

  otherwise

     printf("*W* mfs_print: unknown class ""%s""\n", class);

  end    % Class

end

function itm = getitem(item, cmp)

# Input  item    Item to be processed
#        cmp     Structure with component data
#
# Output itm     Name of item
#
# The function resolves the item definition. Currently, selectors are
# not supported for aerodynamic components.
#
# ---------------------------------------------------------------------

  if (iscell(item))
     itm = item{1};
     printf("*W* mfs_print: selectors are not supported for ");
     printf("aerodynamic components\n");
  else
     itm = item;
  end

end
