function panres = mfs_results_vlm(panels, statresp, q);

# usage: panres = mfs_results_vlm(panels, statresp, q);
#
# Input  panels      Structure with panel data
#        statresp    Structure with static response
#        q(:)        Array with dynamic pressure
#
# Output panres      Structure with panel results
#
# Structure panres has the following fields:
#    nofpan              Number of panels
#    P(3, nofpan)        Coordinates of point between A and B
#    f(3, nofpan, nconf) Panel force vectors
#    p(nofpan, nconf)    Panel pressure
#
# --------------------------------------------------------------------

  warning("off", "Octave:broadcast");

# Check arguments

  if (nargin != 3 || nargout != 1)
     print_usage();
  end

# Number of panels and configurations

  nofpan = panels.nofpan;
  nconf  = statresp.nconf;

# Coordinates of points the aerodynamic force is applied to

  P = 0.5 * (panels.A + panels.B);

# Forces and pressure

  p = zeros(nofpan, nconf);

  rAB = panels.B - panels.A;
  fn  = [zeros(1, nofpan); -rAB(3, :); rAB(2, :)] ;
  for n = 1 : nconf
      f(:, :, n) = 2 * q(n) * fn .* statresp.G(:, n)';
      for m = 1 : nofpan
          p(m, n) = panels.nvec(:, m)' * f(:, m, n) / panels.area(m);
      end
  end

# Build structure with results

  panres = struct("nofpan", nofpan, "P", P, "f", f, "p", p);

  warning("on", "Octave:broadcast");

end
