function [cmp, rc] = mfs_resultsx(cmp, class, items)

# usage:  [cmp, rc] = mfs_resultsx(cmp, class, items)
#
# Input  cmp        Structure with component
#        class      Result class
#        items{}    List of items
#
# Output cmp        Structure with component
#
# The function computes the secondary results of vortex lattice models.
#
# List of results:
#
# class = statresp
#    panres    Panel results
#
# class = trim
#    panres    Panel results
#
# class = freqresp
#    resp.panres  Panel results
#
# Fields of structure panres:
#    nofpan             Number of panels
#    P(3, nofpan)       Coordinates of point between A and B 
#    f(3, nofpan, :)    Panel force vectors
#    p(nofpan, :)       Panel pressure
#
# --------------------------------------------------------------------

# Check arguments

  if (nargin < 3 || nargout != 2)
     print_usage();
  end

# Initialize

  rc  = 1;

# Switch according to class and item

  switch class

  case "statresp"

     if (! isfield(cmp, "statresp"))
        printf("*W* There are no primary results ");
        printf("of class \"statresp\"\n");
        return;
     end

     for r = 1 : length(items)
         switch items{r}
         case "panel"
            cmp.statresp.panres = ...
              mfs_results_vlm(cmp.panels, cmp.statresp,
                              cmp.config.qdyn);
         otherwise
            printf("*E* Unknown result request \"%s\"\n", items{r});
           return;
         end
     end

  case "trim"

     if (! isfield(cmp, "trim"))
        printf("*W* There are no primary results of class \"trim\"\n");
        return;
     end

     for r = 1 : length(items)
         switch items{r}
         case "panel"
            cmp.trim.panres = ...
              mfs_results_vlm(cmp.panels, cmp.trim, ...
                              cmp.config.qdyn);
         otherwise
            printf("*E* Unknown result request \"%s\"\n", items{r});
           return;
         end
     end

  case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*W* There are no primary results ");
        printf("of class \"freqresp\"\n");
        return;
     end

     for l = 1 : cmp.freqresp.nfa     

         resp = cmp.freqresp.resp{l};

         if (! isempty(resp))

            for r = 1 : length(items)
                switch items{r}
                case "panel"
                    panres = mfs_results_vlmh(cmp.panels, cmp.pancols,
                                              cmp.cref, resp);
                    cmp.freqresp.resp{l}.panres = panres;
                otherwise
                   printf("*E* Unknown result request \"%s\"\n", items{r});
                  return;
                end
            end

         end

     end

  otherwise
     printf("*E* Unknown result class \"%s\"\n", class);
     return;

  end

  rc = 0;

end
