function [out, rc] = mfs_xydatax(cmp, class, item, args)

# usage: [out, rc] = mfs_xydatax(cmp, class, item, args)
#
# Input  cmp        Structure with component data
#        class      Response class
#        item       Response item
#        args{:}    Additional arguments depending on class and item
#
# Output out{:}     Output arguments depending on class and item
#        rc         Return code: 0 = no errors
#                                1 = errors
#
# This function returns data for xy-plots of an aerodynamic analysis.
#
# Response classes and items:
#
#  "statresp"   Steady areodynamics
#
#    item        Meaning
#    ----------------------------------------------------------------------
#    "pressure"  Aerodynamic pressure
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  cfg(:)      List of configuration numbers
#                                     (optional; default is all 
#                                      configurations)
#                out{1}   x(:)        x-coordinates
#                out{2}   p(:, :)     Pressure (columns correspond to
#                                     configurations)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#    "vortex"    Vortex strength
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  cfg(:)      List of configuration numbers
#                                     (optional; default is all 
#                                      configurations)
#                out{1}   x(:)        x-coordinates
#                out{2}   G(:, :)     Vortex strength (columns correspond 
#                                     to configurations)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#
#  "trim"       Trim analysis
#
#    item        Meaning
#    ----------------------------------------------------------------------
#    "pressure"  Aerodynamic pressure
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  cfg(:)      List of configuration numbers
#                                     (optional; default is all 
#                                      configurations)
#                out{1}   x(:)        x-coordinates
#                out{2}   p(:, :)     Pressure (columns correspond to
#                                     configurations)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#    "vortex"    Vortex strength
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  cfg(:)      List of configuration numbers
#                                     (optional; default is all 
#                                      configurations)
#                out{1}   x(:)        x-coordinates
#                out{2}   G(:, :)     Vortex strength (columns correspond 
#                                     to configurations)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#
#  "freqresp"   Unsteady harmonic areodynamics
#
#    item        Meaning
#    ----------------------------------------------------------------------
#    "pressure"  Aerodynamic pressure
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  kred(:)     List of reduced frequency indices
#                                     (optional; default is all)
#                args{4}  ldc         Load case number
#                                     (optional; default: 1)
#                out{1}   x(:)        x-coordinates
#                out{2}   p(:, :)     Pressure (columns correspond to
#                                     reduced frequencies)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#    "vortex"    Vortex strength
#                args{1}  lsids(:)    List of lifting surface identifiers
#                args{2}  ycolno      Panel column number
#                args{3}  kred(:)     List of reduced frequency indices
#                                     (optional; default is all)
#                args{4}  ldc         Load case number
#                                     (optional; default: 1)
#                out{1}   x(:)        x-coordinates
#                out{2}   p(:, :)     Vortex strength (columns correspond
#                                     to reduced frequencies)
#                out{3}   y           y-coordinate (optional)
#                out{4}   z           z-coordinate (optional)
#                out{5}   pids(:)     Panel identifiers (optional)
#
# Remarks:
#
#  1. Panel columns are in x-direction.
#  2. Lifting surfaces must be aligned in x-direction.
#
# -------------------------------------------------------------------------

# Check arguments

  if (nargin < 3 || nargout < 2)
     print_usage();
  end

# Initialize

  nargs = length(args);
  out   = cell();
  rc    = 1;

# Branch according to response class

  switch class

  case "statresp"

     if (! isfield(cmp, "statresp"))
        printf("*E* mfs_xydata: no static response data found\n");
        return;
     end

     switch item

     case "pressure"           % Aerodynamic pressure

        lsids  = args{1};
        ycolno = args{2};

        if (nargs == 2 || isempty(args{3}))
           cfg = 1 : cmp.statresp.nconf;
        else
           cfg = args{3};
           if ((cfb = max(cfg)) > cmp.statresp.nconf || 
               (cfb = min(cfg)) < 1)
              printf("*E* mfs_xydata: configuration %d ", cfb);
              printf("does not exist\n");
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = cmp.statresp.panres.P(1, pids);
        out{2} = cmp.statresp.panres.p(pids, cfg);
        out{3} = cmp.statresp.panres.P(2, pids(1));
        out{4} = cmp.statresp.panres.P(3, pids(1));
        out{5} = pids;

     case "vortex"             % Vortex strength

        lsids  = args{1};
        ycolno = args{2};

        if (nargs == 2 || isempty(args{3}))
           cfg = 1 : cmp.statresp.nconf;
        else
           cfg = args{3};
           if ((cfb = max(cfg)) > cmp.statresp.nconf || 
               (cfb = min(cfg)) < 1)
              printf("*E* mfs_xydata: configuration %d ", cfb);
              printf("does not exist\n");
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = 0.5 * (cmp.panels.A(1, pids) + cmp.panels.B(1, pids));
        out{2} = cmp.statresp.G(pids, cfg);
        out{3} = 0.5 * (cmp.panels.A(2, pids(1)) + cmp.panels.B(2, pids(1)));
        out{4} = 0.5 * (cmp.panels.A(3, pids(1)) + cmp.panels.B(3, pids(1)));
        out{5} = pids;

     otherwise

        printf("*E* mfs_xydata: unknown response item \"%s\"\n", item);

     end

  case "trim"

     if (! isfield(cmp, "trim"))
        printf("*E* mfs_xydata: no trim results found\n");
        return;
     end

     switch item

     case "pressure"           % Aerodynamic pressure

        lsids  = args{1};
        ycolno = args{2};

        if (nargs == 2 || isempty(args{3}))
           cfg = 1 : cmp.trim.nconf;
        else
           cfg = args{3};
           if ((cfb = max(cfg)) > cmp.trim.nconf || 
               (cfb = min(cfg)) < 1)
              printf("*E* mfs_xydata: configuration %d ", cfb);
              printf("does not exist\n");
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = cmp.trim.panres.P(1, pids);
        out{2} = cmp.trim.panres.p(pids, cfg);
        out{3} = cmp.trim.panres.P(2, pids(1));
        out{4} = cmp.trim.panres.P(3, pids(1));
        out{5} = pids;

     case "vortex"             % Vortex strength

        lsids  = args{1};
        ycolno = args{2};

        if (nargs == 2 || isempty(args{3}))
           cfg = 1 : cmp.trim.nconf;
        else
           cfg = args{3};
           if ((cfb = max(cfg)) > cmp.trim.nconf || 
               (cfb = min(cfg)) < 1)
              printf("*E* mfs_xydata: configuration %d ", cfb);
              printf("does not exist\n");
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = 0.5 * (cmp.panels.A(1, pids) + cmp.panels.B(1, pids));
        out{2} = cmp.trim.G(pids, cfg);
        out{3} = 0.5 * (cmp.panels.A(2, pids(1)) + cmp.panels.B(2, pids(1)));
        out{4} = 0.5 * (cmp.panels.A(3, pids(1)) + cmp.panels.B(3, pids(1)));
        out{5} = pids;

     otherwise

        printf("*E* mfs_xydata: unknown response item \"%s\"\n", item);

     end

  case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*E* mfs_xydata: no frequency response results found\n");
        return;
     end

     switch item

     case "pressure"           % Aerodynamic pressure

        lsids  = args{1};
        ycolno = args{2};

        if (nargs < 4 || isempty(args{4}))
           lc = 1;
        else
           lc = args{4};
           if (lc < 1 || lc > cmp.freqresp.nfa)
              printf("*E* mfs_xydata: loadcase %d does not exist\n", lc);
              return;
           end
        end

        resp = cmp.freqresp.resp{lc};

        if (nargs < 3 || isempty(args{3}))
           kred = 1 : resp.nfreq;
        else
           kred = args{3};
           if ((kb = max(kred)) > resp.nfreq || 
               (kb = min(kred)) < 1)
              printf("*E* mfs_xydata: reduced frequency index ");
              printf("(%d) out of range\n", kb);
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = resp.panres.P(1, pids);
        out{2} = resp.panres.p(pids, kred);
        out{3} = resp.panres.P(2, pids(1));
        out{4} = resp.panres.P(3, pids(1));
        out{5} = pids;

     case "vortex"             % Vortex strength

        lsids  = args{1};
        ycolno = args{2};

        if (nargs < 4 || isempty(args{4}))
           lc = 1;
        else
           lc = args{4};
           if (lc < 1 || lc > cmp.freqresp.nfa)
              printf("*E* mfs_xydata: loadcase %d does not exist\n", lc);
              return;
           end
        end

        resp = cmp.freqresp.resp{lc};

        if (nargs < 3 || isempty(args{3}))
           kred = 1 : resp.nfreq;
        else
           kred = args{3};
           if ((kb = max(kred)) > resp.nfreq || 
               (kb = min(kred)) < 1)
              printf("*E* mfs_xydata: reduced frequency index ");
              printf("(%d) out of range\n", kb);
           end
        end

        [pids, rc] = mfs_findpanelcol(cmp.ls, lsids, ycolno);
        if (rc) return; end

        out{1} = resp.panres.P(1, pids);
        out{2} = resp.G(pids, kred);
        out{3} = resp.panres.P(2, pids(1));
        out{4} = resp.panres.P(3, pids(1));
        out{5} = pids;

     otherwise

        printf("*E* mfs_xydata: unknown response item \"%s\"\n", item);

     end

  otherwise

     printf("*E* mfs_xydata: unknown response class \"%s\"\n", class);

  end

end
