function [out, rc] = mfs_getrespc(cmp, class, item, args)

# usage: [out, rc] = mfs_getrespc(cmp, class, item, args)
#
# Input  cmp        Structure with component data
#        class      Response class
#        item       Response item
#        args{:}    Additional arguments depending on class and item
#
# Output out{:}     Output arguments depending on class and item
#        rc         Return code: 0 = no errors
#                                1 = errors
#
# This functions returns responses of an aeroelastic analysis.
#
# Response classes and items:
#
#  "flutter"    Flutter anaylsis
#
#    item        Meaning
#    ----------------------------------------------------------------------
#    "curves"    Flutter curves
#                args{1}  modno(:) Flutter mode numbers
#                                  (optional; default is all modes)
#                out{1}   v(:, :)  Velocities
#                out{2}   g(:, :)  Loss factors or
#                         a(:, :)  Real parts of eigenvalues
#                out{3}   k(:, :)  Reduced frequencies
#                out{4}   f(:, :)  Frequencies
#
# Remarks:
#
#  1. Flutter curves:
#      a) Matrix columns correspond to flutter modes.
#      b) Matrix rows correspond to reduced frequencies (k-method) or
#         velocities (pk-method).
#      c) For the k-method, matrix k contains one column only.
#      d) For the pk-method, matrix pk contains one column only.
#
# -------------------------------------------------------------------------

# Check arguments

  if (nargin != 4 || nargout != 2)
     print_usage();
  end

# Process variable arguments

  nargs = length(args);
  out   = cell(2);

# Initialize

  rc = 0;

# Branch according to response class

  switch class

# Flutter

  case "flutter"

     if (! isfield(cmp, "flutter"))
        printf("*E* mfs_getresp: no flutter results found\n");
        rc = 1;
        return;
     end

#    Process item

     switch item

     case "curves"
        if (nargs == 0 || isempty(args{1}))
           modix = 1 : cmp.flutter.nofmod;
        else
           modix = lookup(cmp.flutter.modno, args{1}, "m");
           ixbad = find(modix == 0);
           if (! isempty(ixbad))
              for n = 1 : length(ixbad)
                printf("*E* mfs_getresp: flutter mode %3.0d does not exist\n",
                       args{1}(ixbad(n)));
              end
              rc = 1; return;
           end
        end

        switch cmp.flutter.method
        case "k"
           out{1} = cmp.flutter.v(modix, :)';
           out{2} = cmp.flutter.g(modix, :)';
           out{3} = cmp.flutter.kred';
        case "pk"
           out{1} = cmp.flutter.v';
           out{2} = cmp.flutter.a(modix, :)';
           out{3} = cmp.flutter.kred(modix, :)';
        end
        out{4} = cmp.flutter.f(modix, :)';

     otherwise

        printf("*E* mfs_getresp: unknown response item \"%s\"\n", item);
        rc = 1;

     end   % item

  otherwise

     printf("mfs_getresp: unknown response class \"%s\"\n", class);
     rc = 1;

  end       % class

end
