function [load, rc] = mfs_new_load(msg, inload, controls, panels, cref);

# usage: [load, rc] = mfs_new_load(msg, inload, controls, panels, cref);
#
# Input  msg         File handle of message file
#        inload      Structure with input load data
#        controls    Structure with control surface data
#        panels      Structure with panel data
#        cref        Reference chord length
#
# Output load        Structure with loads
#        rc          Return code: 0 means no erros
#                                 1 means errors
#
# The function builds the load structure.
#
# Input data:
#
#   inload.gust(:)   Gust loads
#
# Output data:
#
#   load             see Programmer's Manual
#
# -----------------------------------------------------------------------------

  typelist = {"gust", "manoeuvre"};  

# Check arguments

  if (nargin != 5 || nargout != 2)
     print_usage();
  end

# Initialize

  load = struct();

# Check fields

  if ((rc = mfs_checkfields4(msg, inload, typelist, "loads")))
     return;
  end

# Get number of loadcases

  nofldc = 0;

  for [val, name] = inload
      if (isfield(val, "lc"))
         nofldc = max([nofldc, [val.lc]]);
      else
         nofldc = max(nofldc, 1);
      end
  end

# Initialize load information

  inflc  = zeros(nofldc, 1);
  isgust = 0;
  ismnvr = 0;

# Process load types

  for [val, name] = inload

      switch name
      case "gust"
         [gust, inflc, rcg] = mfs_new_load_gust(msg, val, nofldc, inflc);
         rc = rc | rcg;
         isgust = 1;
      case "manoeuvre"
         [mnvr, inflc, rcm] = mfs_new_load_mnvr(msg, val, controls, panels,
                                                nofldc, inflc, cref);
         rc = rc | rcm;
         ismnvr = 1;
      end

  end 

  bad = find(inflc == 3);
  if (! isempty(bad))
     fprintf(msg, "*E* loads: gusts and manoeuvres cannot be mixed ");
     fprintf(msg, "in one loadcase\n");
     rc = 1;
  end

  if (rc) return; end

# Build the load structure

  load = struct("nofldc", nofldc, "inflc", inflc);
  if (isgust)
     load = setfield(load, "gust", gust);
  end
  if (ismnvr)
     load = setfield(load, "mnvr", mnvr);
  end

end
