function [gust, inflc, rc] = mfs_new_load_gust(msg, gustdef, nofldc, inflc);

# usage: [gust, inflc, rc] = mfs_new_load_gust(msg, gustdef, nofldc, inflc);
#
# Input  msg           File handle of message file
#        gustdef(:)    Structure array with gust definitions
#        inflc(:)      Load case information
#
# Output gust(nofldc)  Structure array with gust data
#        inflc(:)      Load case information 
#                      (Bit 1 indicates gust loads)
# Output load          Structure with loads
#        rc            Return code: 0 means no erros
#                                   1 means errors
#
# The function checks the definitions of the gust loads and builds
# structure gust with the gust data:
#
#  gust(nofldc).qdyn   Dynamic pressure
#              .v      Flight velocity
#              .wg     Gust velocity
#              .x0     x-coordinate of reference point
#
# --------------------------------------------------------------------

  legal_fields = {"qdyn", "wg", "v", "x0", "lc"};
  req_fields   = {"qdyn", "wg", "v"};

# Check arguments

  if (nargin != 4 || nargout != 3)
     print_usage();
  end

# Initialize

  gust  = struct();
  lcdef = zeros(1, nofldc);

# Check fields

  if ((rc = mfs_checkfields1(msg, gustdef, req_fields,
                             "loads.gust")))
     return;
  end

  if ((rc = mfs_checkfields4(msg, gustdef, legal_fields,
                             "loads.gust")))
     return;
  end

# Check data

  ndef = length(gustdef);

  for n = 1 : ndef

      qdyn = gustdef(n).qdyn;
      if (isempty(qdyn))
         fprintf(msg, "*E* loads.gust(%d): qdyn undefined\n", n);
         rc = 1;
      elseif (! isscalar(qdyn))
         fprintf(msg, "*E* loads.gust(%d): qdyn must be a scalar\n", n);
         rc = 1;
      elseif (qdyn <= 0)
         fprintf(msg, "*E* loads.gust(%d): qdyn not positive\n", n);
         rc = 1;
      end            

      wg = gustdef(n).wg;
      if (isempty(wg))
         fprintf(msg, "*E* loads.gust(%d): wg undefined\n", n);
         rc = 1;
      elseif (! isscalar(wg))
         fprintf(msg, "*E* loads.gust(%d): wg must be a scalar\n", n);
         rc = 1;
      end

      v = gustdef(n).v;
      if (isempty(v))
         fprintf(msg, "*E* loads.gust(%d): v undefined\n", n);
         rc = 1;
      elseif (! isscalar(v))
         fprintf(msg, "*E* loads.gust(%d): v must be a scalar\n", n);
         rc = 1;
      end

      if (! isfield(gustdef, "x0") || isempty(gustdef(n).x0))
         gustdef(n).x0 = 0;
      end

      if (! isfield(gustdef, "lc") || isempty(gustdef(n).lc))
         lc = 1; gustdef(n).lc = 1;
         inflc(1) = bitset(inflc(1), 1);
      else
         if ((lc = gustdef(n).lc) < 1)
            fprintf(msg, "*E* loads.gust(%d): bad definition of", n);
            fprintf(msg, " loadcase number\n");
            rc = 1;
         else
            inflc(lc) = bitset(inflc(lc), 1);
         end
      end
      if (lcdef(lc))
         fprintf(msg, "*E* loads.gust(%d): loadcase %d ", n, lc)
         fprintf(msg, "alread defined\n");
         rc = 1;
      else
         lcdef(lc) = 1;
      end 

  end

  if (rc) return; end

# Build output structure

  for n = 1 : ndef
      lc = gustdef(n).lc;
      gust(lc) = struct("qdyn", gustdef(n).qdyn,
                        "v"   , gustdef(n).v,
                        "wg"  , gustdef(n).wg,
                        "x0"  , gustdef(n).x0); 
  end

end
