function [cmp, rc] = mfs_newc(msg, model, opts)

# usage: [cmp, rc] = mfs_newc(msg, model)
#
# Input  msg     File handle of message file
#        model   Structure with model data
#        opts    Structure with options 
#
# Output cmp     Structure with component data
#                (0 if errors occurred)
#        rc      Return code: 0 = no errors
#                             1 = errors
#
# The function initializes a new aeroelastic component.
# The following fields are defined:
#
#  type         Component type
#  solid        Structure with data of solid component
#  aero         Structure with data of aerodynamic compnent
#  splines      Structure with spline data
#  load         Structure with load data
#
# --------------------------------------------------------------------

  legal_fields = {"type", "solid", "aero", "splines", "loads"};
  req_fields   = {"type", "solid", "aero", "splines"};

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Initialize

  cmp = struct();

# Check fields

  if ((rc = mfs_checkfields1(msg, model, req_fields,
                             "Model definition")))
     return;
  end

  if ((rc = mfs_checkfields4(msg, model, legal_fields,
                             "Model definition")))
     return;
  end

# Define type

  fprintf(msg, "\n  Model Type = %s\n\n", model.type);

  cmp = struct("type", "aeroelastic");

# Load the solid component

  rc = 1;

  if (! isstruct(model.solid))
     fprintf(msg, "*E* %s.solid is not a structure\n",
             inputname(2));
     return;
  end

  if (! isfield(model.solid, "type"))
     fprintf(msg, "*E* %s.solid is not a component\n",
             inputname(2));
     return
  end
  if (! strcmp(model.solid.type, "solid"))
     fprintf(msg, "*E* %s.solid is not a solid component\n",
             inputname(2));
     return;
  end
  if (! strcmp(model.solid.subtype, "3d"))
     fprintf(msg, "*E* Only 3d solids are supported");
     fprintf(msg, " in aeroelasticity\n");
     return;
  end
  if (! isfield(model.solid, "stiff"))
     fprintf(msg, "*E* Solid component has no stiffness matrix\n");
     return;
  end

  cmp.solid = struct("type",     model.solid.type,
                     "subtype",  model.solid.subtype,
                     "nodes",    model.solid.nodes,
                     "elements", model.solid.elements,
                     "dofs",     model.solid.dofs,
                     "nset",     model.solid.nset,
                     "eset",     model.solid.eset,
                     "stiff",    model.solid.stiff);

  if (isfield(model.solid, "mass"))
     cmp.solid = setfield(cmp.solid, "mass", model.solid.mass);
  end
  if (isfield(model.solid, "modes"))
     cmp.solid = setfield(cmp.solid, "modes", model.solid.modes);
  end
  if (isfield(model.solid, "damping"))
     cmp.solid = setfield(cmp.solid, "damping", model.solid.damping);
  end
                     
# Load the aerodynamic component

  if (! isstruct(model.aero))
     fprintf(msg, "*E* %s.aero is not a structure\n",
             inputname(2));
     return;
  end

  if (! isfield(model.aero, "type"))
     fprintf(msg, "*E* %s.aero is not a component\n",
             inputname(2));
     return
  end
  if (! strcmp(model.aero.type, "aero"))
     fprintf(msg, "*E* %s.aero is not an aerodynamic component\n",
             inputname(2));
     return;
  end

  cmp.aero = struct("type",     model.aero.type,
                    "subtype",  model.aero.subtype,
                    "cref",     model.aero.cref,
                    "nodes",    model.aero.nodes,
                    "ls",       model.aero.ls,
                    "panels",   model.aero.panels,
                    "pancols",  model.aero.pancols,
                    "controls", model.aero.controls);

  if (isfield(model.aero, "symy"))
     cmp.aero = setfield(cmp.aero, "symy", model.aero.symy);
  end

  if (isfield(model.aero, "config"))
     cmp.aero = setfield(cmp.aero, "config", model.aero.config);
  end

# Splines

  [splines, rc] = mfs_new_splines(msg, cmp.solid, cmp.aero,
                                  model.splines, opts.ptol);
  if (rc) return; end

  cmp = setfield(cmp, "splines", splines);

# Loads

  if (isfield(model, "loads"))
     [load, rc] = mfs_new_load(msg, model.loads, cmp.aero.controls,
                               cmp.aero.panels, cmp.aero.cref);
     if (rc)
        return;
     else
        cmp = setfield(cmp, "load", load);
     end
  end

# Print messages

  fprintf(msg, "  Number of splines      = %3d\n", cmp.splines.nofspl);
  if (isfield(cmp, "load"))
     fprintf(msg, "  Number of load cases   =  %2d\n",
             cmp.load.nofldc);
  end

end
