function [cmpt, rc] = mfs_trf(cmp, cmpi, class, item)

# usage: [cmpt, rc] = mfs_trf(cmp, cmpi, class, item)
#
# Input  cmp      Stucture with data of aeroelastic component
#        cmpi     Structure with data of input component
#        class    String identifying class
#        item     String identifying item to be transferred
#
# Output cmpt     Structure with data of target component
#
# The function tranfers data from the input component (solid or
# aero) to the target component (aero or solid).
#
# List of classes:
#
#   aero => solid:  statresp, trim
#   solid => aero:  statresp, modes
#
# List of items:
#
#    Target    Item     Description
# --------------------------------------------------------------
#    solid     loads    Aerodynamic loads are transferred to the
#                       solid component.
#    aero      disp     Displacements are transferred to the
#                       aerodynamic component.
#
# --------------------------------------------------------------------

# Check arguments

  if (nargin != 4 || nargout != 2)
     print_usage();
  end

# Initialize

  cmpt = struct();
  rc   = 1;

# Check availability of splines

  if (! isfield(cmp, "splines"))
     printf("*E* mfs_transfer: definition of splines missing\n");
     return;
  end
  if (! isfield(cmp.splines, "Shg"))
     printf("*E* mfs_transfer: spline matrices missing\n");
     return;
  end

# Branch according to type of input component, class and item

  switch cmpi.type

  case "aero"      % Transfer from aero to solid component

     cmpt = cmp.solid;

     switch class

     case "statresp"

        if (! isfield(cmpi, "statresp"))
           printf("*E* mfs_transfer: aerodynamic results missing\n");
           return;
        end

        switch item

        case "loads"
           cmpt = mfs_trf_loads(cmpt, cmpi, cmpi.statresp, 
                                cmp.splines.Shg, cmp.splines.Svh);
           rc = 0;
        otherwise
           printf("*E* mfs_transfer: unknown item \"%s\"\n", item);
           return

        end

     case "trim"

        if (! isfield(cmpi, "trim"))
           printf("*E* mfs_transfer: aerodynamic results missing\n");
           return;
        end

        switch item

        case "loads"
           cmpt = mfs_trf_loads(cmpt, cmpi, cmpi.trim,
                                cmp.splines.Shg, cmp.splines.Svh);
           rc = 0;
        otherwise
           printf("*E* mfs_transfer: unknown item \"%s\"\n", item);
           return

        end

     otherwise

        printf("*E* mfs_transfer: unknown class \"%s\"\n", class);

     end

  case "solid"     % Transfer from solid to aero component

     cmpt = cmp.aero;

     switch class

     case "statresp"

        if (! isfield(cmpi, "statresp"))
           printf("*E* mfs_transfer: static displacements missing\n");
           return;
        end

        switch item

        case "disp"
           cmpt = mfs_trf_disp(cmpt, cmpi.statresp,
                          cmp.splines.Shg, cmp.splines.Snh);
           rc = 0;
        otherwise
           printf("*E* mfs_transfer: unknown item \"%s\"\n", item);
           return

        end

     case "modes"

        if (! isfield(cmpi, "modes"))
           printf("*E* mfs_transfer: normal modes missing\n");
           return;
        end

        switch item

        case "disp"
           cmpt = mfs_trf_modes(cmpt, cmpi.modes, cmpi.stiff.ndofr,
                          cmp.splines.Shg, cmp.splines.Snh);
           rc = 0;
        otherwise
           printf("*E* mfs_transfer: unknown item \"%s\"\n", item);
           return

        end

     otherwise

        printf("*E* mfs_transfer: type \"%s\" ", cmpi.type);
        printf("of input component not supported\n");
        return; 

     end

  end

end
