function [model, rc] = mfs_importx(msg, tdata, fname)

# usage: [model, rc] = mfs_importx(msg, tdata, fname)
#
# Input  msg      File handle of message file
#        tdata    Structure with translation data
#        fname    Name of input file
#
# Output model    Structure wit model definition
#        rc       Return code: 0 = no errors, 1 = errors found
#
# The function imports a solid model from an m-file defining a finite element
# mesh for CIMNE MAT-fem (see http://www.cimne.com/mat-fem/).
#
# -----------------------------------------------------------------------------

  rc = 1;

# Model type and subtype

  model.type = tdata.type;

  if (! isfield(tdata, "subtype"))
     fprintf(msg, "*E* mfs_import: model subtype undefined\n");
     return;
  else
     model.subtype = tdata.subtype;
  end

# Run the m-file defining the MAT-fem data

  try
     source(fname);
  catch
     fprintf(msg, "*E* mfs_import: could not process file %s\n",
             fname);
     return;
  end

# Geometry and material

  geom.t = thick;

  mat.type = "iso";
  mat.E    = young;
  mat.ny   = poiss;
  mat.rho  = denss;

# Coordinates

  np = rows(coordinates);
  for n = 1 : np
      nodes(n).id = n;
      nodes(n).coor = coordinates(n, :);
  end

  model.nodes = nodes;

# Elements

  [ne, npe] = size(elements);
  if (npe == 4)
     etype = "s4";
  elseif (npe == 3)
     etype = "s3";
  else
     fprintf(msg, "*E* mfs_import: only elements with 3 or 4 nodes");
     fprintf(msg, " supported\n");
     return;
  end

  for n = 1 : ne
      elem(n).id = n; elem(n).type = etype;
      elem(n).nodes = elements(n, :);
      elem(n).geom = geom; elem(n).mat = mat;
  end

  model.elements = elem;

# Constraints

  nc = rows(fixnodes);
  for n = 1 : nc
      fix(n).id   = fixnodes(n, 1);
      fix(n).dofs = fixnodes(n, 2);
  end

  model.constraints.prescribed = fix;

# Point loads

  np = rows(pointload);
  n  = 1; m = 1;
  data = zeros(1, 6);
  idl = pointload(1, 1); dof = pointload(1, 2);
  data(dof) = pointload(1, 3);

  while (++n <= np)
     if((id = pointload(n, 1)) != idl)
        force(m).id = idl; force(m++).data = data; 
        data = zeros(1, 6);
        idl  = id;
     end
     dof = pointload(n, 2);
     data(dof) = pointload(n, 3);
  end

  force(m).id = idl; force(m).data = data;

  model.loads.point = force;

  rc = 0;

end

