function sig = mfs_beamstress(fid, cmp, class, eltsel, coor, ldc)

# usage: sig = mfs_beamstress(fid, cmp, class, eltsel, coor, ldc)
#
# Input  fid             File handle of output file
#        cmp             Structure with component data
#        class           Response class
#        eltsel          Element selector
#        coor(:, 2)      Coordinates of stress points
#        ldc(:)          Loadcase list (optional; 
#                                       default: all loadcases)
#
# Output sig(:, :, :)    Bending stresses (optional):
#                         1st index: elements
#                         2nd index: stresspoints
#                         3rd index: loadcases
#
# The function computes the normal stresses of beams.
#
# Response classes supported:
#
#    statresp     Static response
#    trim         Trim analysis
#
# ---------------------------------------------------------------------

# Copyright (c) 2024 by Johannes Wandinger

  t0 = clock();

  sig = [];

# Check arguments

  if (nargin < 5 || nargin > 6 || nargout > 1)
     print_usage();
  endif
  if (! is_valid_file_id(fid))
     error("mfs_beamstress: first argument does not refer to an open file\n");
  endif
  if (! isstruct(cmp))
     fclose(fid);
     error("mfs_beamstress: second argument is not a structure\n");
  endif
  if (! ischar(class))
     fclose(fid);
     error("mfs_beamstress: third argument is not a string\n");
  endif

  [np, nc] = size(coor);

# Check component type

  if (! strcmp(cmp.type, "solid"))
     printf("mfs_beamstress: model type \"%s\" not supported\n",
            cmp.type);
  endif

# Check availability of results

  switch class
  case {"statresp", "trim"}
     if (! isfield(cmp, class))
        printf("mfs_beamstress: there are no results of class %s\n",
               class);
        return;
     endif
     cdata = getfield(cmp, class);
     if (! isfield(cdata, "elres"))
        printf("mfs_beamstress: there are no element results\n");
        return;
     endif
     if (nargin < 6)
        switch class
        case "statresp"
           ldc = 1 : cdata.nofldc;
        case "trim"
           ldc = 1 : cdata.nconf;
        endswitch 
     endif
  otherwise
     printf("mfs_beamstress: class \"%s\" not supported\n",
            class);
     return;
  endswitch

# Header

  fprintf(fid, "\n");
  for k = 1 : 8
      fprintf(fid, "----------");
  end
  fprintf(fid, "\n\nComponent ""%s""\n\n", inputname(2));

  time_stamp = ignore_function_time_stamp("all");
  mfs_paths("add", "mfs_beamstress.m", "solid");

# Get element indices

  [ixelt, rc] = mfs_findelts(cmp.elements.index, eltsel, cmp.eset);

  if (! rc)

# Process elements

     switch cmp.subtype
     case "2d"
        if (nc == 1)
           sig = mfs_beamstress2d(fid, cmp, cdata, ixelt, coor, ldc);
        else
           printf("*E* mfs_beamstress: ");
           printf("number of stress point coordinates is not 1\n");
        endif
     case "3d"
        if (nc == 2)
           sig = mfs_beamstress3d(fid, cmp, cdata, ixelt, coor, ldc);
        else
           printf("*E* mfs_beamstress: ");
           printf("number of stress point coordinates is not 2\n");
        endif
     endswitch

  endif

  mfs_paths("remove");
  ignore_function_time_stamp(time_stamp);

# End

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to compute beam stresses of component %s\n",
         elapsed_time, inputname(2));
  fflush(stdout);

endfunction
