function mfs_export(fname, format, cmp, class, varargin)

# usage: mfs_export(fname, format, cmp, class, varargin)
#
# Input  fname     Name of output file
#        format    Format of output file
#        cmp       Structure with component
#        class     Output data class
#        varargin  Output requests
#
# The function exports data to a file with given format.
#
# Supported formats:
#
#   msh     Gmsh mesh file
#   matfem  m-file for CIMNE MAT-fem (no output requests)
#
# Output requests for solid components:
#
#  class = mesh    Output the mesh
#        mesh      Nodal points and elements (default)
#        axes      Unit vectors of the element coordinate
#                  systems
#
#  class = load
#        point     Point loads
#
#  class = statresp
#        disp      Displacements
#        rot       Rotations
#        reac      Reaction loads
#        stress    Stresses
#        resultant Stress resultants
#
#  class = modes
#        disp      Displacements
#        rot       Rotations
#
#  class = freqresp
#        disp      Displacements
#        resultant Stress resultants
#        stress    Stresses
#
#  class = transresp
#        disp      Displacements
#        resultant Stress resultants
#        stress    Stresses
#
#  class = diverg
#        disp      Displacements
#
#  class = trim
#        disp      Displacements
#        stress    Stresses
#        resultant Stress resultants
#
# Output requests for aerodynamic components:
#
#  class = mesh    Output the mesh
#        camber    Include the camber 
#
#  class = statresp
#        pressure  Aerodynamic pressure
#        force     Aerodynamic forces
#        disp      Displacements
#
#  class = diverg
#        disp      Displacements
#
#  class = trim
#        pressure  Aerodynamic pressure
#        force     Aerodynamic forces
#        disp      Displacements
#
#  class = modes
#        disp      Displacements
#
# -----------------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger

  t0 = clock();

# Supported component types

  supported_types = {"solid", "aero"};

# Check arguments

  if (nargin < 3)
     print_usage();
  end

  if (! ischar(fname))
     error("mfs_export: first input argument must be a file name\n");
  end
  if (! ischar(format))
     error("mfs_export: second input argument must be a file format\n");
  end
  if (! isstruct(cmp))
     error("mfs_export: third input argument must be a structure\n");
  end
  if (! ismember(cmp.type, supported_types))
     error("mfs_export: component type \"%s\" not supported\n", cmp.type);
  end

# Branch according to file format

  switch format

  case {"msh", "msh22", "msh41"}
     if (nargin < 4)
        print_usage();
     end
     if ((fid = fopen(fname, "wt")) == -1)
        error("mfs_export: could not open file \"%s\"\n", fname);
     end
     switch format
     case "msh22"
        vers = "2.2";
     case {"msh", "msh41"}
        vers = "4.1";
     end
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_export.m", ["msh/", cmp.type],
               ["msh/", cmp.type, "/", vers],
               ["msh/util/", vers]);
     mfs_exportx(fid, vers, cmp, class, varargin);
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);
     fclose(fid);

  case "matfem"
     if (nargin > 3)
        print_usage();
     end
     if ((fid = fopen(fname, "wt")) == -1)
        error("mfs_export: could not open file \"%s\"\n", fname);
     end
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_export.m", "matfem");
     mfs_exportx(fid, cmp);
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);
     fclose(fid);

  otherwise
     error("mfs_export: File format %s not supported\n", format);      

  end

# End

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to write file %s\n", ...
          elapsed_time, fname);

  fflush(stdout);

end
