function nodes = mfs_linenodes(nodes, id1, id2, idnew, bias)

# usage:  nodes = mfs_linenodes(nodes, id1, id2, idnew, bias)
#
# Input  nodes      Structure array with nodal point data
#        id1        Identifier of first node
#        id2        Identifier of last node
#        idnew      Identifiers of new nodes
#        bias       Bias (optional)
#
# Output nodes      Structure array with nodal point data
#
# The function adds nodes along a line between the existing nodes with
# identifiers id1 and id2 to the structure array with nodal point data.
#
# The bias is the ratio between the length of the last interval and
# the first interval. If bias is not given, the nodes are equally spaced.
#
# ------------------------------------------------------------------------

# Copyright (c) 2015 by Johannes Wandinger

# Check arguments

  if (nargin < 4 || nargin > 5 || nargout != 1)
     print_usage();
  endif

# Determine number of nodes to be created

  nofgen = length(idnew);

# Create parametric positions

  if (nargin == 4)
     x = linspace(-1, 1, nofgen + 2);
     r = x(2 : nofgen + 1);
  else
     a = 1;
     b = a + (nofgen + 1) * log10(bias) / nofgen;
     x = logspace(a, b, nofgen + 2);
     r = -1 + 2 * (x(2 : nofgen + 1) - x(1)) / (x(nofgen + 2) - x(1));
  end

# Find positions of given nodes

  nofnod = length(nodes);

  ix1 = 1;
  ix2 = 1;

  while(ix1 <= nofnod && nodes(ix1).id != id1)
     ix1++;
  end
  if(ix1 > nofnod)
    error("mfs_linenodes: node %5d does not exist\n", id1);
  end

  while(ix2 <= nofnod && nodes(ix2).id != id2)
     ix2++;
  end
  if(ix2 > nofnod)
    error("mfs_linenodes: node %5d does not exist\n", id2);
  end

# Create nodes

  ix = nofnod + 1;
  for n = 1 : nofgen
      nodes(ix).id = idnew(n);
      nodes(ix).coor  = (1 - r(n)) * nodes(ix1).coor;
      nodes(ix).coor  = 0.5 * (nodes(ix).coor + (1 + r(n)) * nodes(ix2).coor);
      ix++;
  end

end
