function cmp = mfs_mass(cmp, type)

# usage: cmp = mfs_mass(cmp, type)
#
# Input  cmp     Structure with component data
#        type    Mass type: "lumped" (default) or "consistent"
#
# Output cmp     Structure with component data 
#                (mass matrix added)
#
# The function computes the mass matrix of a component.
#
# ---------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger

  t0 = clock();

# Check arguments

  if (nargin < 1 || nargin > 2 || nargout != 1)
     print_usage();
  end
  if (! isstruct(cmp))
     error("mfs_mass: argument must be a structure\n");
  end

  if (nargin == 2)
     if (ischar(type))
        switch type
        case "lumped"
           mtype = 1;
           fctnam = "mfs_mle";
        case "consistent"
           mtype = 2;
           fctnam = "mfs_mce";
        otherwise
           error("mfs_mass: unknown mass type ""%s""\n", type);
        end
     else
        error("mfs_mass: second argument must be a string\n");
     end
  else
     mtype = 1;
     fctnam = "mfs_mle";
  end

  if (! strcmp(cmp.type, "solid"))
     error("msf_mass: component type must be \"solid\"\n");
  end

  time_stamp = ignore_function_time_stamp("all");

# Manage paths

  mfs_paths("add", "mfs_mass.m", "solid", "solid/elem/util");

  bpath = file_in_loadpath("mfs_stiff.m");
  len   = length(bpath) - length("mfs_stiff.m");
  bpath = strcat(bpath(1 : len), "solid/elem/", cmp.subtype, "/");

# Initialize the mass matrix

  ndofg = cmp.dofs.ndofg;
  M     = sparse(ndofg, ndofg);

# Element loop to assemble mass matrix

  elem = cmp.elements.elem;

  for l = 1 : cmp.elements.noftyp

      epath = strcat(bpath, cmp.elements.types(l).name);
      addpath(epath);

      if (exist(fctnam, "file") == 2)

         efun = str2func(fctnam);

         m1 = cmp.elements.types(l).ixelt1;
         m2 = m1 + cmp.elements.types(l).nofelt - 1;

         for m = m1 : m2
             ect = elem(m).ects;
             M(ect, ect) = M(ect, ect) + efun(elem(m)); 
         end

     end

     rmpath(epath);

  end

# End

  mfs_paths("remove");

  mass     = struct("type", mtype, "M", M);
  cmp.mass = mass;

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to assemble %5d by %5d mass matrix\n", ...
         elapsed_time, ndofg, ndofg);
  fflush(stdout);

  ignore_function_time_stamp(time_stamp);

end
