function cmp = mfs_new(fid, model, options)

# usage: cmp = mfs_new(fid, model, options)
#
# Input  fid       File handle of message file  
#        model     Structure with model data
#        options   Structure with options (optional)
#
# Output cmp    Structure with component data
#               (0 if errors occurred)
#
# The function initializes a new component.
#
# -----------------------------------------------------------------------------

# Copyright (c) by 2020 Johannes Wandinger

# Initialize

  t0  = clock();
  cmp = struct();

  isopts = 0;

# Check arguments

  if (nargin < 2 || nargin > 3 || nargout != 1 )
     print_usage();
  end

  if (! is_valid_file_id(fid))
     error("mfs_new: first argument does not refer to an open file\n");
  end
  if (! isstruct(model))
     fclose(fid);
     error("mfs_new: second argument must be a structure\n");
  end

# Look for options

  if (nargin == 3)
     isopts = 1;
     if (! isstruct(options))
        fclose(fid);
        error("mfs_new: third input argument must be a structure\n");
     end
  end

# Print version

  fprintf(fid, "Mefisto %s: ", mfs_version());
  fprintf(fid, "Building new component from input ""%s""\n", inputname(2));

# Branch on model type

  if (isfield(model, "type"))

     switch model.type

     case {"solid", "structure"}
        model.type = "solid";
        time_stamp = ignore_function_time_stamp("all");
        mfs_paths("add", "mfs_new.m", "solid", "util",
                  "solid/elem/util");
        defopts = struct("chkelts",  1);
        if (isopts)
           opts = mfs_defopts(options, defopts);
        else
           opts = defopts;
        end
        [cmp, rc] = mfs_newx(fid, model, opts);
        mfs_paths("remove");
        ignore_function_time_stamp(time_stamp);

     case "aero"
        time_stamp = ignore_function_time_stamp("all");
        mfs_paths("add", "mfs_new.m", "aero", "util");
        defopts = struct("rtolp", 1e-4);
        if (isopts)
           opts = mfs_defopts(options, defopts);
        else
           opts = defopts;
        end
        [cmp, rc] = mfs_newx(fid, model, opts);
        mfs_paths("remove");
        ignore_function_time_stamp(time_stamp);

     case "aeroelastic"
        time_stamp = ignore_function_time_stamp("all");
        mfs_paths("add", "mfs_new.m", "aeroelastic", "util");
        defopts = struct("ptol", 1e-4);
        if (isopts)
           opts = mfs_defopts(options, defopts);
        else
           opts = defopts;
        end
        [cmp, rc] = mfs_newc(fid, model, opts);
        mfs_paths("remove"); 
        ignore_function_time_stamp(time_stamp);

     otherwise
        fclose(fid);
        error("mfs_new: unknown model type %s\n", model.type);
     end 

  else
     fclose(fid);
     error("mfs_new: model type not defined\n");
  end

  fprintf(fid, "\n");

# End

  if (rc)
     text   = sprintf("mfs_new: Could not create component from input \"%s\"",
                      inputname(2));
     mfs_errexit(text, fid);
  else
     elapsed_time = etime(clock(), t0);
     printf("%10.4f seconds needed to build new component from input %s\n", ... 
            elapsed_time, inputname(2));
  end

  fflush(stdout);

end
