function varargout = mfs_statresp(cmp, opts)

# usage: varargout = mfs_statresp(cmp, opts)
#
# Input  cmp         Structure with component data
#        opts        Structure with options (optional)
#
# Output varargout   Output data
#
# The function computes the static response of a component.
#
# The output data depend on the component type:
#
#  - solid:         varargout = cmp
#  - aerodynamic:   varargout = cmp
#  - aeroelastic:   varargout = [cmps, cmpa]
#
#    cmps    Structure with data of solid component
#    cmpa    Structure with data of aerodynamic data
#
# ---------------------------------------------------------------------

# Copyright (c) 2016 by Johannes Wandinger

  t0 = clock();

# Check arguments

  if (nargin > 2 || nargout > 2)
     print_usage();
  end
  if (! isstruct(cmp))
     error("mfs_statresp: input argument must be a structure\n");
  end

# Branch on model type

  switch cmp.type

  case "solid"
     if (nargout != 1)
        print_usage();
     end
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_statresp.m", "solid", "util");
     [cmp, rc] = mfs_statrespx(cmp);
     varargout{1} = cmp;
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);

  case "aero"
     if (nargout != 1)
        print_usage();
     end
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_statresp.m", "aero", "util");
     [cmp, rc] = mfs_statrespx(cmp);
     varargout{1} = cmp;
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);

  case "aeroelastic"
     if (nargout != 2)
        print_usage();
     end
     defopts = struct("method", "direct", "tol", 1e-6, 
                      "maxit", 20);
     if (nargin < 2)
        opts = defopts;
     else
        opts = mfs_defopts(opts, defopts);
     end
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_new.m", "aeroelastic", "aero", "util");
     [cmps, cmpa, rc] = mfs_statrespc(cmp, opts);
     varargout = {cmps, cmpa};
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);

  otherwise
     error("mfs_statresp: model type \"%s\" not supported\n", cmp.type);
  end

  if (rc)
     error("mfs_statresp ended with errors\n");
  end

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to compute static response of component %s\n",
         elapsed_time, inputname(1));
  fflush(stdout);

end
