function cmpt = mfs_transfer(cmp, cmpi, class, item)

# usage: cmpt = mfs_transfer(cmp, cmpi, class, item)
#
# Input  cmp      Structure with data of aeroelastic component
#        cmpi     Structure with data of input component
#        class    String identifying class
#        item     String identifying item to be transferred
#
# Output cmpt     Structure with data of target component
#
# The function tranfers data from the input component (solid or
# aero) to the target component (aero or solid).
#
# List of classes:
#
#   aero => solid:  statresp, trim
#   solid => aero:  statresp, modes 
#
# List of items:
#
#    Target    Item     Description
# --------------------------------------------------------------
#    solid     loads    Aerodynamic loads are transferred to the
#                       solid component.
#    aero      disp     Displacements are transferred to the
#                       aerodynamic component.
#
# This function can be used to check the correct definition of the
# splines.
#
# --------------------------------------------------------------------

# Copyright (c) 2017 by Johannes Wandinger

  t0 = clock();

# Check arguments

  if (nargin != 4 || nargout != 1)
     print_usage();
  end
  if (! isstruct(cmp))
     error("mfs_transfer: first argument must be a structure\n");
  end
  if (! isstruct(cmpi))
     error("mfs_transfer: second argument must be a structure\n");
  end
  if (! ischar(class))
     error("mfs_transfer: third argument must be a string\n");
  end
  if (! ischar(item))
     error("mfs_transfer: fourth argument must be a string\n");
  end

  cmpt = struct();

# Branch according type

  switch cmp.type
  case "aeroelastic"
     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_transfer.m", "aeroelastic", "util");
     [cmpt, rc] = mfs_trf(cmp, cmpi, class, item);
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);
     if (rc)
        error("mfs_transfer: could not transfer data\n");
     end
  otherwise
     error("mfs_transfer: component type \"%s\" not supported\n",
           cmp.type);
  end

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to transfer data\n",
         elapsed_time);
  fflush(stdout);

end
