function [cmp, t] = mfs_transresp(cmp, tdef, varargin)

# usage: cmp = mfs_transresp(cmp, tdef, varargin)
#
# Input  cmp       Structure with component data
#        tdef(:)   Array with definition of time
#        varargin  Parameters
#
# Output cmp       Structure with component data
#                  (transient response added)
#        t(:)      Array with time steps (optional)
#
# Parameters
#
#   Name       Values    Default   Meaning
#  -----------------------------------------------------------------------
#   method     enhanced  enhanced  Enhanced modal reduction
#              modal               Modal reduction
#   load       load(:)   none      Structure array defining loads
#   rcase      integer     1       Result case number
#   statresp     lc      none      Number of static load case defining
#                                  initial conditions
#   transresp  [rc, tp]  none      Number of transient result case and
#                                  time step number defining initial
#                                  conditions
#   param      [a, b]  [0.25, 0.5] Parameters of Newmark algorithm
#
# Fields of structure load
#
#   lc          Load case defining load pattern
#   func        Function defining time function
#   params(:)   Parameters to be supplied to func
#
# The function computes the transient response. Currently, only solid
# structures are supported.
#
# ------------------------------------------------------------------------

# Copyright (c) 2021 by Johannes Wandinger

  t0 = clock();

# Check arguments

  if (nargin < 2 || nargout < 1 || nargout > 2)
     print_usage();
  end
  if (! isstruct(cmp))
     error("mfs_transresp: %s must be a structure\n", inputname(1));
  end
  if (length(tdef) != 2)
     error("mfs_transresp: %s must define time step and simulation time\n",
           inputname(2));
  elseif (tdef(1) <= 0 || tdef(2) <= 0)
     error("mfs_transresp: bad definition of %s\n", inputname(2));
  elseif (tdef(1) > tdef(2))
     error("mfs_transresp: bad definition of time\n");
  end

# Branch on component type

  switch cmp.type

  case "solid"

     time_stamp = ignore_function_time_stamp("all");
     mfs_paths("add", "mfs_transresp.m", "solid", "util");
     [cmp, t, rc] = mfs_transrespx(cmp, tdef, varargin);
     mfs_paths("remove");
     ignore_function_time_stamp(time_stamp);

  otherwise
     error("mfs_transresp: model type \"%s\" not supported\n",
           cmp.type);

  end

  if (rc)
     error("mfs_transresp ended with errors\n");
  end

  elapsed_time = etime(clock(), t0);
  printf("%10.4f seconds needed to process %5d time steps of component %s\n", ...
         elapsed_time, length(t), inputname(1));
  fflush(stdout);
 
end
