function mfs_exportx(fid, vers, cmp, class, items)

# usage: mfs_exportx(fid, vers, cmp, class, items)
#
# Input  fid       File identifier of output file
#        vers      File format version
#        cmp       Structure with component
#        class     Output data class
#        items     List of items
#
# The function writes the requested items to a file in Gmsh MSH format.
#
# ------------------------------------------------------------------------

# Write the file header

  fprintf(fid, "$MeshFormat\n");
  fprintf(fid, "%s 0%2d\n", vers, sizeof(1));
  fprintf(fid, "$EndMeshFormat\n");

# Process parameters

  switch class

  case "mesh"

     mshflag = 1;

     if (length(items) == 0)
        mfs_exp_mesh(fid, cmp);
     else
        if (ismember("mesh", items) && ismember("camber", items))
           printf("*W* mfs_export: items ""mesh"" and ""camber"" found\n");
           printf("                ignoring item ""mesh""\n");
           msflag = 0;
        end
        for item = items
            switch item{1}
            case "mesh"
                if (mshflag)
                   mfs_exp_mesh(fid, cmp);
                end
            case "camber"
                cmp.nodes = mfs_camber(cmp);
                mfs_exp_mesh(fid, cmp);
            case "normals"
                mfs_exp_mesh_normals(fid, cmp);
            otherwise
                printf("*W* mfs_export: unknown item ""%s""\n", item{1});
            end
        end
     end

  case "statresp"

     if (! isfield(cmp, "statresp"))
        printf("*W* mfs_export: there are no items of class \"statresp\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "pressure"
            for n = 1 : cmp.statresp.nconf
                title{n} = sprintf("%s: Pressure", cmp.config.names{n});
            end
            mfs_exp_panel_scalar(fid, cmp.statresp.panres, ...
                                 cmp.statresp.panres.p, title);
         case "force"
            for n = 1 : cmp.statresp.nconf
                title{n} = sprintf("%s: Force", cmp.config.names{n});
            end
            mfs_exp_panel_vector(fid, cmp.statresp.panres, ...
                                 cmp.statresp.panres.f, title);
         case "disp"
            for n = 1 : cmp.statresp.nconf
                title{n} = sprintf("%s: Displ.", cmp.config.names{n});
            end
            if (isfield(cmp.statresp, "disp"))
               mfs_exp_disp(fid, cmp, cmp.statresp.disp, title);
            else
               printf("*W* mfs_export: no displacements found\n");
            end
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  case "diverg"

     if (! isfield(cmp, "diverg"))
        printf("*W* mfs_export: there are no items of class \"diverg\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for m = 1 : cmp.diverg.ndiv
                title{m} = sprintf("Divergence shape %2d: qdyn = %12.4e", 
                                    m, cmp.diverg.qdyn(m));
            end
            mfs_exp_disp(fid, cmp, cmp.diverg.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item ""%s""\n", item{1});
         end         % item
     end             % item loop

  case "trim"

     if (! isfield(cmp, "trim"))
        printf("*W* mfs_export: there are no items of class \"trim\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "pressure"
            for n = 1 : cmp.trim.nconf
                title{n} = sprintf("%s: Pressure", cmp.config.names{n});
            end
            mfs_exp_panel_scalar(fid, cmp.trim.panres, ...
                                 cmp.trim.panres.p, title);
         case "force"
            for n = 1 : cmp.trim.nconf
                title{n} = sprintf("%s: Force", cmp.config.names{n});
            end
            mfs_exp_panel_vector(fid, cmp.trim.panres, ...
                                 cmp.trim.panres.f, title);
         case "disp"
            for n = 1 : cmp.trim.nconf
                title{n} = sprintf("%s: Displ.", cmp.config.names{n});
            end
            if (isfield(cmp.trim, "disp"))
               mfs_exp_disp(fid, cmp, cmp.trim.disp, title);
            else
               printf("*W* mfs_export: no displacements found\n");
            end
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  case "modes"

     if (! isfield(cmp, "modes"))
        printf("*W* mfs_export: there are no items of class ""modes""\n");
        return;
     end
 
     for item = items
         switch item{1}
         case "disp"
            for m = 1 : cmp.modes.nofmod
                title{m} = sprintf("Mode shape %3d: %10.3f Hz", 
                                    m, cmp.modes.freq(m));
            end
            mfs_exp_disp(fid, cmp, cmp.modes.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
      end

 case "flutter"

     if (! isfield(cmp, "flutter"))
        printf("*W* mfs_export: there are no items of class \"flutter\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for n = 1 : cmp.flutter.nshape
               title1 = sprintf("Flutter Shape: Mode = %2d, Point = %2d, ",
                                cmp.flutter.mode(n), cmp.flutter.point(n));
               title2 = sprintf("v = %10.3e, f = %7.2f",
                                cmp.flutter.v(n), cmp.flutter.f(n));
               title{n} = [title1, title2];
            end
            mfs_exp_cdisp(fid, cmp, cmp.flutter.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

 case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*W* mfs_export: there are no items of class \"freqresp\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for l = 1 : cmp.freqresp.nfa
                if (! isempty(resp = cmp.freqresp.resp{l}))
                   if (isfield(resp, "U"))
                      for n = 1 : resp.nback
                          title{n} = sprintf("LC %3d: Deformation at %10.3f Hz",
                                             l, resp.freqback(n));
                      end
                      mfs_exp_cdisp(fid, cmp, resp.U, title);
                   end
                end
            end
         case "pressure"
            for l = 1 : cmp.freqresp.nfa
                if (! isempty(resp = cmp.freqresp.resp{l}))
                   if (isfield(resp, "panres"))
                      if (isfield(resp, "freq"))
                         freq = resp.freq;
                         for n = 1 : length(freq)
                             title{n} = sprintf("LC %3d: Pressure at f = %7.2f Hz",
                                                l, freq(n));
                         end
                      else
                         kred = resp.kred;
                         for n = 1 : length(kred)
                             title{n} = sprintf("LC %3d: Pressure at k = %7.4f",
                                                l, kred(n));
                         end
                      end
                      mfs_exp_panel_cscalar(fid, resp.panres, resp.panres.p,
                                            title); 
                   end
                end 
            end
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  otherwise
 
     printf("*W* mfs_export: unknown class ""%s""\n", class);

  end                % class

end
