function [nodes, rc] = mfs_imp_nodes(msg, fid, subtype)

# usage: [nodes, rc] = mfs_imp_nodes(msg, fid, subtype)
#
# Input  msg       File handle of message file
#        fid       Input file handle
#        subtype   Model subtype
#
# Output nodes     Structure array with nodal point data
#        rc        Return code: 0 = no errors, 1 = errors found
#
# The function reads the nodal point data from the input file (version 2.2
# file format).
#
# -----------------------------------------------------------------------------

  nodes = struct();
  rc    = 1;

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  end

# Get problem dimension

  switch subtype
  case "2d"
     ncoor = 2;
  case "3d"
     ncoor = 3;
  otherwise
     fprintf(msg, "*E* Unknown model subtype ""%s"" found\n", subtype);
     return;
  end
  
# Header record

  line = fgetl(fid);
  if (! strcmp(line, "$Nodes"))
     fprintf(msg, "*E* $Nodes expected but not found\n");
     return;
  end

# Number of nodes

  line = fgetl(fid);
  nofnod = sscanf(line, "%d", 1);
  if (nofnod < 1)
     fprintf(msg, "*E* Number of nodes (%d) not positive\n", nofnod);
     return;
  end

# Nodal point data

  for n = 1 : nofnod
      line  = fgetl(fid);
      words = strsplit(line);
      nodes(n).id   = sscanf(words{1}, "%d");
      for k = 1 : ncoor
          nodes(n).coor(k) = sscanf(words{k + 1}, "%f");
      end
  end

# End record

  line = fgetl(fid);
  if (! strcmp(line, "$EndNodes"))
     fprintf(msg, "*E* $EndNodes expected but not found\n");
     return;
  end

  rc = 0;

end
