function [entities, rc] = mfs_imp_entities(msg, fid)

# usage: [entities, rc] = mfs_imp_entities(msg, fid)
#
# Input  msg       File handle of message file
#        fid       Input file handle
#
# Output entites   Cell array with entity tables
#                  {1} point entity table
#                  {2} curve entity table
#                  {3} surface entity table
#                  {4} volume entity table
#        rc        Return code: 0 = no errors, 1 = errors found
#
# The function returns the entity tables. The entity tables are structure
# arrays containg the fields
#     table(:).entity      entity tag
#     table(:).numphys     number of physical tags
#     table(:).phystag(:)  physical tags
# The tables are sorted according to ascending entity tags.
#
# -----------------------------------------------------------------------------

  entities = cell(4, 1);
  rc       = 1;

# Check arguments

  if (nargin != 2 || nargout != 2)
     print_usage();
  end

# Header record

  line = fgetl(fid);
  if (! strcmp(line, "$Entities"))
     fprintf(msg, "*E* $Entities expected but not found\n");
  end

# Number of entities

  line = fgetl(fid);
  nume = sscanf(line, "%d", 4);

# Process point entities

  etable = struct();

  for n = 1 : nume(1)

      line  = fgetl(fid);
      words = strsplit(line);

      pointTag = sscanf(words{1}, "%d");
      numphys  = sscanf(words{5}, "%d");

      etable(n).entity  = pointTag;
      etable(n).numphys = numphys;
      if (numphys)
         physTag = zeros(1, numphys);
         for k = 1 : numphys
             physTag(k) = sscanf(words{5 + k}, "%d");
         end
         etable(n).phystag = physTag;
      end

  end

  if (nume(1))
     [ids, ix] = sort([etable.entity]);
     etable    = etable(ix);
  end

  entities{1} = etable;

# Process curve entities

  etable = struct();

  for n = 1 : nume(2)

      line  = fgetl(fid);
      words = strsplit(line);

      curveTag = sscanf(words{1}, "%d");
      numphys  = sscanf(words{8}, "%d");

      etable(n).entity  = curveTag;
      etable(n).numphys = numphys;
      if (numphys)
         physTag = zeros(1, numphys);
         for k = 1 : numphys
             physTag(k) = sscanf(words{8 + k}, "%d");
         end
         etable(n).phystag = physTag;
      end

  end

  if (nume(2))
     [ids, ix] = sort([etable.entity]);
     etable    = etable(ix);
  end

  entities{2} = etable;

# Process surface entities

  etable = struct();

  for n = 1 : nume(3)

      line  = fgetl(fid);
      words = strsplit(line);

      surfTag = sscanf(words{1}, "%d");
      numphys = sscanf(words{8}, "%d");

      etable(n).entity  = surfTag;
      etable(n).numphys = numphys;
      if (numphys)
         physTag = zeros(1, numphys);
         for k = 1 : numphys
             physTag(k) = sscanf(words{8 + k}, "%d");
         end
         etable(n).phystag = physTag;
      end

  end

  if (nume(3))
     [ids, ix] = sort([etable.entity]);
     etable    = etable(ix);
  end

  entities{3} = etable;

# Process volume entities

  etable = struct();

  for n = 1 : nume(4)

      line  = fgetl(fid);
      words = strsplit(line);

      volTag  = sscanf(words{1}, "%d");
      numphys = sscanf(words{8}, "%d");

      etable(n).entity  = volTag;
      etable(n).numphys = numphys;
      if (numphys)
         physTag = zeros(1, numphys);
         for k = 1 : numphys
             physTag(k) = sscanf(words{8 + k}, "%d");
         end
         etable(n).phystag = physTag;
      end

  end

  if (nume(4))
     [ids, ix] = sort([etable.entity]);
     etable    = etable(ix);
  end

  entities{4} = etable;

# End record

  line = fgetl(fid);
  if (! strcmp(line, "$EndEntities"))
     fprintf(msg, "*E* $EndEntities expected but not found\n");
     return;
  end

  rc = 0;

end
