function [nodes, rc] = mfs_imp_nodes(msg, fid, subtype)

# usage: [nodes, rc] = mfs_imp_nodes(msg, fid, subtype)
#
# Input  msg       File handle of message file
#        fid       Input file handle
#        subtype   Model subtype
#
# Output nodes     Structure array with nodal point data
#        rc        Return code: 0 = no errors, 1 = errors found
#
# The function reads the nodal point data from the input file (version 4.1
# file format).
#
# -----------------------------------------------------------------------------

  nodes = struct();
  rc    = 1;

# Check arguments

  if (nargin != 3 || nargout != 2)
     print_usage();
  endif

# Get problem dimension

  switch subtype
  case "2d"
     ncoor = 2;
  case "3d"
     ncoor = 3;
  otherwise
     fprintf(msg, "*E* Unknown model subtype \"%s\" found\n", subtype);
     return;
  endswitch
  
# Header record

  line = fgetl(fid);
  while (! strcmp(line, "$Nodes"))
     mfs_skip(fid);
     line = fgetl(fid);
     if (line == -1)
        fprintf(msg, "*E* $Nodes expected but not found\n");
        return;
     endif
  endwhile

# Number of blocks and nodes

  [numBlocks, numNodes, minNodeTag, maxNodeTag] = ...
    fscanf(fid, "%d %d %d %d\n", "C");
  if (numNodes < 1)
     fprintf(msg, "*E* Number of nodes (%d) not positive\n", nofnod);
     return;
  endif

# Process blocks

  n2 = 0;

  for nB = 1 : numBlocks

      line  = fgetl(fid);
      words = strsplit(line);
      np    = sscanf(words{4}, "%d"); % Number of nodes in block

      if (np)
         nodeTag = fscanf(fid, "%d\n", np);
         ids     = num2cell(nodeTag);
         xyz     = fscanf(fid, "%f\n", [3, np]);
         coor    = mat2cell(xyz(1 : ncoor, :)', ones(np, 1), ncoor);
         n1      = n2 + 1; 
         n2     += np;
         nodes(n1 : n2) = struct("id", ids, "coor", coor);
      endif

  endfor

# End record

  line = fgetl(fid);
  if (! strcmp(line, "$EndNodes"))
     fprintf(msg, "*E* $EndNodes expected but not found\n");
     return;
  endif

  rc = 0;

endfunction
