function [model, nodesets, elsets, rc] = mfs_importx(msg, fid, tdata, fname)

# usage: [model, nodesets, elsets, rc] = mfs_importx(msg, fid,  tdata, fname)
#
# Input  msg      File handle of message file
#        fid      File handle of input file
#        tdata    Structure with translation data
#        fname    Name of input file
#
# Output model    Structure with model definition
#        nodesets Structure with node sets
#        elsets   Structure with element sets
#        rc       Return code: 0 = no errors, 1 = errors found
#
# The function reads a Gmsh msh-file using the 4.1 file version.
#
# -----------------------------------------------------------------------------

  model    = struct();
  nodesets = struct();
  elsets   = struct();

# Check arguments

  if (nargin != 4 || nargout != 4)
     print_usage();
  end

# Check input data

  if ((rc = mfs_imp_check(msg, tdata)))
     return;
  end

# Read the physical names

  [physnam, tdata, rc] = mfs_imp_physical(msg, fid, tdata);
  if (rc) return; end

# Read the entities

  [entities, rc] = mfs_imp_entities(msg, fid);
  if (rc) return; end

# Read the nodes

  [nodes, rc] = mfs_imp_nodes(msg, fid, tdata.subtype);
  if (rc) return; end

# Read the elements

  [elements, loads, constraints, nodesets, elsets, rc] = ...
     mfs_imp_elements(msg, fid, physnam, entities, tdata);
  if (rc) return; end

# Build the structure

  model = struct("type", tdata.type, "subtype", tdata.subtype, ...
                 "nodes", nodes, "elements", elements,
                 "nset", nodesets, "eset", elsets);
  if (isstruct(loads))
     model = setfield(model, "loads", loads);
  end
  if (isstruct(constraints))
     model = setfield(model, "constraints", constraints);
  end
  if (isfield(tdata, "damping"))
     model = setfield(model, "damping", tdata.damping);
  end

end
