function mfs_exp_cstress3d(fid, elres, title)

# usage: mfs_exp_cstress3d(fid, elres, title)

# Input  fid         File handle of output file
#        elres       Structure with element results
#        title(:)    Cell array with view names
#
# The function exports complex stresses of 3-dimensional elements in 
# Gmsh MSH file format. The real part is written as time step 0 and
# the imaginary part as time step 1.
#
# --------------------------------------------------------------------

  zero = 0;

# Define output formats

  f3e = " %-10.3e %-10.3e %-10.3e";
  fst = cstrcat("%-5d", f3e, f3e, f3e, "\n");

# Title modifications for shells

  suffix = {" (Upper Surface)", " (Lower Surface)"};

# Get number of elements with stresses

  nofelt  = 0;
  isshell = 0;

  for ntype = 1 : elres.noftyp
      if (elres.types(ntype).rtype > 1)
         nofelt += elres.types(ntype).nofelt;
         isshell = isshell || elres.types(ntype).rtype == 4;
      end
  end

  if (isshell)
     pass = 2;
  else
     pass = 1;
  end

# Loop over frequencies

  for l = 1 : elres.nofcol

      for p = 1 : pass

#         Real Part

%         Header

          fprintf(fid, "$ElementData\n");

%         Tags

          if (isshell)
             fprintf(fid, "1\n""%s""\n", [title{l}, suffix{p}]);
          else
             fprintf(fid, "1\n""%s""\n", title{l});
          end
          fprintf(fid, "1\n%-10.3f\n", zero);
          fprintf(fid, "3\n%-3d\n9\n%-5d\n", 0, nofelt);

          for ntype = 1 : elres.noftyp

              nelt = elres.types(ntype).nofelt;
              m1   = elres.types(ntype).ixelt1;
              m2   = m1 + nelt - 1;

              switch elres.types(ntype).rtype

                case 2  % rod
                  R = zeros(1, 9);
                  for m = m1 : m2
                      R(1) = real(elres.elem(m).results.sig(l));
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

                case 3  % plane stress
                  for m = m1 : m2
                      T = elres.elem(m).results.TE;
                      sig1 = real(elres.elem(m).results.sig(:, l, :));
                      sig  = squeeze(sig1);
                      R = mfs_sig2dto3d(T, sig, elres.elem(m).nofpnt);
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

                case 4  % shell
                  for m = m1 : m2
                      T = elres.elem(m).results.TE;
                      sig = real(elres.elem(m).results.sig(:, l, p));
                      R = mfs_sig2dto3d(T, sig, elres.elem(m).nofpnt);
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

              end

          end   % Loop over element types

#         End

          fprintf(fid, "$EndElementData\n");

#         Imaginary Part

%         Header

          fprintf(fid, "$ElementData\n");

%         Tags

          if (isshell)
             fprintf(fid, "1\n""%s""\n", [title{l}, suffix{p}]);
          else
             fprintf(fid, "1\n""%s""\n", title{l});
          end
          fprintf(fid, "1\n%-10.3f\n", zero);
          fprintf(fid, "3\n%-3d\n9\n%-5d\n", 1, nofelt);

          for ntype = 1 : elres.noftyp

              nelt = elres.types(ntype).nofelt;
              m1   = elres.types(ntype).ixelt1;
              m2   = m1 + nelt - 1;

              switch elres.types(ntype).rtype

                case 2  % rod
                  R = zeros(1, 9);
                  for m = m1 : m2
                      R(1) = imag(elres.elem(m).results.sig(l));
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

                case 3  % plane stress
                  for m = m1 : m2
                      T = elres.elem(m).results.TE;
                      sig1 = imag(elres.elem(m).results.sig(:, l, :));
                      sig  = squeeze(sig1);
                      R = mfs_sig2dto3d(T, sig, elres.elem(m).nofpnt);
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

                case 4  % shell
                  for m = m1 : m2
                      T = elres.elem(m).results.TE;
                      sig = imag(elres.elem(m).results.sig(:, l, p));
                      R = mfs_sig2dto3d(T, sig, elres.elem(m).nofpnt);
                      fprintf(fid, fst, elres.elem(m).id, R);
                  end

              end

          end   % Loop over element types

#         End

          fprintf(fid, "$EndElementData\n");

      end       % Shell surface loop

  end           % Loadcase loop

end
