function mfs_exportx(fid, vers, cmp, class, items)

# usage: mfs_exportx(fid, vers, cmp, class, items)
#
# Input  fid       File identifier of output file
#        vers      File format version
#        cmp       Structure with component
#        class     Output data class
#        items     List of items
#
# The function writes the requested items to a file in Gmsh MSH format.
#
# ------------------------------------------------------------------------

# Write the file header

  fprintf(fid, "$MeshFormat\n");
  fprintf(fid, "%s 0%2d\n", vers, sizeof(1));
  fprintf(fid, "$EndMeshFormat\n");

# Process parameters

  switch class

  case "mesh"

     if (length(items) == 0)
        mfs_exp_mesh(fid, cmp);
     else
        for item = items
            switch item{1}
            case "mesh"
                mfs_exp_mesh(fid, cmp);
            case {"axes", "axesonly"}
                mfs_exp_mesh_axes(fid, cmp);
            otherwise
                printf("*W* mfs_export: unknown item ""%s""\n", item{1});
            end
        end
     end

  case "load"

     if (! isfield(cmp, "load"))
        printf("*W* mfs_export: there are no items of class ""load""\n");
        return;
     end

     for item = items
         switch item{1}
         case "point"
            if (! isfield(cmp.load, "point"))
               printf("*W* mfs_export: item ""point"" does not exist\n");
               return;
            end
            for l = 1 : cmp.load.nofldc
                title{l, 1} = sprintf("LC %3d: Applied Force", l);
                title{l, 2} = sprintf("LC %3d: Applied Moment", l);
            end
            mfs_exp_vect2(fid, cmp, cmp.load.point, title);
         otherwise
            printf("*W* mfs_export: unknown item ""%s""\n", item{1});
         end         % item
     end             % item loop


  case "statresp"

     if (! isfield(cmp, "statresp"))
        printf("*W* mfs_export: there are no items of class ""statresp""\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for l = 1 : cmp.statresp.nofldc
                title{l} = sprintf("LC %3d: Deformation", l);
            end
            mfs_exp_disp(fid, cmp, cmp.statresp.disp, title);
         case "rot"
            for l = 1 : cmp.statresp.nofldc
                title{l} = sprintf("LC %3d: Rotations", l);
            end
            mfs_exp_rot(fid, cmp, cmp.statresp.disp, title);
         case "reac"
            for l = 1 : cmp.statresp.nofldc
                title{l, 1} = sprintf("LC %3d: Reac. Force", l);
                title{l, 2} = sprintf("LC %3d: Reac. Moment", l);
            end
            mfs_exp_vect2(fid, cmp, cmp.statresp.reac, title);
         case "resultant"
            for l = 1 : cmp.statresp.nofldc
                title{l} = sprintf("LC %3d: Stress Resultants", l);
            end
            if (isfield(cmp.statresp, "elres"))
               mfs_exp_rslt(fid, cmp.statresp.elres, title);
            else
               printf("*W* mfs_export: no element results found\n");
            end 
         case "stress"
            for l = 1 : cmp.statresp.nofldc
                title{l} = sprintf("LC %3d: Stresses", l);
            end
            if (isfield(cmp.statresp, "elres"))
               mfs_exp_stress(fid, cmp.statresp.elres, title);
            else
               printf("*W* mfs_export: no element results found\n");
            end 
         otherwise
            printf("*W* mfs_export: unknown item ""%s""\n", item{1});
         end         % item
     end             % item loop

  case "modes"

     if (! isfield(cmp, "modes"))
        printf("*W* mfs_export: there are no items of class ""modes""\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for m = 1 : cmp.modes.nofmod
                title{m} = sprintf("Mode shape %3d: %10.3f Hz", 
                                    m, cmp.modes.freq(m));
            end
            mfs_exp_disp(fid, cmp, cmp.modes.disp, title);
         case "rot"
            for m = 1 : cmp.modes.nofmod
                title{m} = sprintf("Mode %3d (Rot.): %10.3f Hz", 
                                    m, cmp.modes.freq(m));
            end
            mfs_exp_rot(fid, cmp, cmp.modes.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item ""%s""\n", item{1});
         end         % item
     end             % item loop

  case "diverg"

     if (! isfield(cmp, "diverg"))
        printf("*W* mfs_export: there are no items of class \"diverg\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for m = 1 : cmp.diverg.ndiv
                title{m} = sprintf("Divergence shape %2d: qdyn = %12.4e", 
                                    m, cmp.diverg.qdyn(m));
            end
            mfs_exp_disp(fid, cmp, cmp.diverg.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item ""%s""\n", item{1});
         end         % item
     end             % item loop

  case "trim"

     if (! isfield(cmp, "trim"))
        printf("*W* mfs_export: there are no items of class \"trim\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for n = 1 : cmp.trim.nconf
                title{n} = sprintf("Configuration %3d: Displacements", n);
            end
            if (isfield(cmp.trim, "disp"))
               mfs_exp_disp(fid, cmp, cmp.trim.disp, title);
            else
               printf("*W* mfs_export: no displacements found\n");
            end
         case "resultant"
            for l = 1 : cmp.trim.nconf
                title{l} = sprintf("Conf. %3d: Stress Resultants", l);
            end
            if (isfield(cmp.trim, "elres"))
               mfs_exp_rslt(fid, cmp.trim.elres, title);
            else
               printf("*W* mfs_export: no element results found\n");
            end 
         case "stress"
            for l = 1 : cmp.trim.nconf
                title{l} = sprintf("Conf. %3d: Stresses", l);
            end
            if (isfield(cmp.trim, "elres"))
               mfs_exp_stress(fid, cmp.trim.elres, title);
            else
               printf("*W* mfs_export: no element results found\n");
            end 
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  case "freqresp"

     if (! isfield(cmp, "freqresp"))
        printf("*W* mfs_export: there are not items of class \"freqresp\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for l = 1 : cmp.freqresp.nfa
                if (! isempty(resp = cmp.freqresp.resp{l}))
                   if (isfield(resp, "U"))
                      for n = 1 : resp.nback
                          title{n} = sprintf("LC %3d: Deformation at %10.3f Hz",
                                             l, resp.freqback(n));
                      end
                      mfs_exp_cdisp(fid, cmp, resp.U, title);
                   end
                end
            end
         case "resultant"
            for l = 1 : cmp.freqresp.nfa
                title1 = sprintf("LC %3d: Stress Resultants at ", l);
                if (isfield(cmp.freqresp.resp{l}, "elres"))
                   elres = cmp.freqresp.resp{l}.elres;
                   fback = cmp.freqresp.resp{l}.freqback;
                   for n = 1 : elres.nofcol
                       title{n} = [title1, sprintf("%10.3f Hz", fback(n))];
                   end
                   mfs_exp_crslt(fid, elres, title);
                else
                   printf("*W* mfs_export: no element results found\n");
                end
            end
         case "stress"
            for l = 1 : cmp.freqresp.nfa
                title1 = sprintf("LC %3d: Stresses at ", l);
                if (isfield(cmp.freqresp.resp{l}, "elres"))
                   elres = cmp.freqresp.resp{l}.elres;
                   fback = cmp.freqresp.resp{l}.freqback;
                   for n = 1 : elres.nofcol
                       title{n} = [title1, sprintf("%10.3f Hz", fback(n))];
                   end
                   mfs_exp_cstress(fid, elres, title);
                else
                   printf("*W* mfs_export: no element results found\n");
                end
            end
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  case "transresp"

     if (! isfield(cmp, "transresp"))
        printf("*W* mfs_export: there are not items of class \"transresp\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for l = 1 : cmp.transresp.nta
                if (! isempty(resp = cmp.transresp.resp{l}))
                   if (isfield(resp, "u"))
                      title = sprintf("LC %3d: Deformation", l);
                      mfs_exp_tdisp(fid, cmp, resp.u, resp.tback, title);
                   end
                end
            end
         case "resultant"
            for l = 1 : cmp.transresp.nta
                if (! isempty(resp = cmp.transresp.resp{l}))
                   if (isfield(resp, "elres"))
                      title = sprintf("LC %3d: Stress Resultants", l); 
                      mfs_exp_trslt(fid, resp.elres, resp.tback, title);
                   end 
                end
            end
         case "stress"
            for l = 1 : cmp.transresp.nta
                if (! isempty(resp = cmp.transresp.resp{l}))
                   if (isfield(resp, "elres"))
                      title = sprintf("LC %3d: Stresses", l); 
                      mfs_exp_tstress(fid, resp.elres, resp.tback, title);
                   end 
                end
            end
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  case "flutter"

     if (! isfield(cmp, "flutter"))
        printf("*W* mfs_export: there are not items of class \"flutter\"\n");
        return;
     end

     for item = items
         switch item{1}
         case "disp"
            for n = 1 : cmp.flutter.nshape
               title1 = sprintf("Flutter Shape: Mode = %2d, Point = %2d, ",
                                cmp.flutter.mode(n), cmp.flutter.point(n));
               title2 = sprintf("v = %10.3e, f = %7.2f",
                                cmp.flutter.v(n), cmp.flutter.f(n));
               title{n} = [title1, title2];
            end
            mfs_exp_cdisp(fid, cmp, cmp.flutter.disp, title);
         otherwise
            printf("*W* mfs_export: unknown item \"%s\"\n", item{1});
         end
     end

  otherwise
 
     printf("*W* mfs_export: unknown class ""%s""\n", class);

  end                % class

end
