function rc = mfs_imp_check(msg, tdata)

# usage: rf = mfs_imp_check(msg, tdata)
#
# Input  tdata   Structure with translation data
#
# Output rc      Return code: 0 = no errors, 1 = errors found
#
# The function checks if the translation data contains the required
# fields.
#
# -----------------------------------------------------------------------------

  rc  = 1;

# Model subtype

  if (! isfield(tdata, "subtype"))
     fprintf(msg, "*E* Model subtype not defined\n");
     return;
  end

# Check the remaining fields

  fields = fieldnames(tdata);
  nf     = length(fields);

  if (nf == 2)
     fprintf(msg, "*E* No physical groups defined in translation data\n");
     return;
  end

  ixtypes = strcmp("type", fields) + strcmp("subtype", fields);
  ixtypes = ixtypes + strcmp("damping", fields);
  ixgroup = find(ixtypes == 0);
  ngroup  = length(ixgroup);

  rc = 0;

  for n = 1 : ngroup
      ixg   = ixgroup(n);
      group = getfield(tdata, fields{ixg});
      if (isstruct(group))
         if (! isfield(group, "type"))
            fprintf(msg, "*E* Type of group %s is not defined\n", ...
                    fields{ixg});
            rc = 1;
         else
            errmsg = sprintf("Group %s", fields{ixg});
            switch group.type
            case "elements"
               if (mfs_checkfields1(msg, group, {"name"}, 
                                    errmsg))
                  rc = 1;
               end
            case "constraints"
               if (mfs_checkfields1(msg, group, {"name"}, errmsg))
                  rc = 1;
               else
                  switch group.name
                  case {"prescribed", "coupled"}
                     if (mfs_checkfields1(msg, group, {"dofs"}, errmsg))
                        rc = 1;
                     elseif ((baddof = min(group.dofs)) < 1 ||
                             (baddof = max(group.dofs)) > 6)
                        fprintf(msg, "%s: illegal dof identifier %d\n",
                                errmsg, baddof);
                        rc = 1;
                     end
                  otherwise
                     fprintf(msg, "*E* Constraint name %s of group %s ", ...
                             group.name, fields{ixg});
                     fprintf(msg, "not supported\n");
                     rc = 1;
                  end
               end
            case "loads"
               if (mfs_checkfields1(msg, group, {"name"}, errmsg))
                  rc = 1;
               else
                  switch group.name
                  case {"point", "disp", "velo", "acce"}
                     if (mfs_checkfields1(msg, group, {"data"}, errmsg))
                        rc = 1;
                     end
                  otherwise
                     fprintf(msg, "*E* Load name %s of group %s ", ...
                             group.name, fields{ixg});
                     fprintf(msg, "not supported\n");
                     rc = 1;
                  end
               end
            case "nodeset"
            otherwise
               fprintf(msg, "*E* Type %s of group %s not supported\n", ...
                       group.type, fields{ixg});
               rc = 1;
            end
         end
      else
         fprintf(msg, "*E* Group %s is not a structure\n", fields{ixg});
         rc = 1;
      end 
  end

end
