function relem = mfs_reselt(nofelt, elem, disp, class, varargin)

# usage: relem = mfs_reselt(nofelt, elem, disp, class, varargin)
#
# Input  nofelt      Number of elements
#        elem(:)     Structure array with element data
#        disp(:, :)  Displacement matrix
#        class       Response class
#
# Output relem(:)    Structure array containing element results
#
# with
#
#        relem.id              Element identifier
#        relem.nofpnt          Number of result points
#        relem.coor(2)         Coordinates of result point
#        relem.rtype           Result type: 2 = rod
#        relem.results.sig(:)  Stress
#        relem.results.eps(:)  Strain
#        relem.results.rslt(:) Normal force
#
# This function computes the results of 2-dimensional r2 elements.
#
# The variable argument list contains additional arguments depending
# on the class:
#
#    statresp:    none
#    freqresp:    varargin{1}      Circular excitation frequency
#
# -------------------------------------------------------------------

# Loop over elements

  for nel = 1 : nofelt

     dx = elem(nel).coor(2, 1) - elem(nel).coor(1, 1);
     dy = elem(nel).coor(2, 2) - elem(nel).coor(1, 2);
     L  = sqrt(dx^2 + dy^2);

     T = [ dx, dy, 0, 0; 0, 0, dx, dy ] / L;

     uE = T * disp(elem(nel).ects, :);

     coor = mean(elem(nel).coor);
     eps  = (uE(2, :) - uE(1, :)) / L;
     sig  = elem(nel).mat.E * eps;
     N    = elem(nel).geom.A * sig;

     results    = struct("sig", sig, "eps", eps, "rslt", N);
     relem(nel) = struct("id", elem(nel).id, "nofpnt", 1, 
                         "coor", coor, "rtype", 2, "results", results);

  end

end
