function relem = mfs_reselt(nofelt, elem, disp, class, varargin)

# usage: relem = mfs_reselt(nofelt, elem, disp, class, varargin)
#
# Input  nofelt      Number of elements 
#        elem(:)     Structure array with element data
#        disp(:, :)  Displacement matrix
#        class       Response class
#
# Output relem(:)    Structure array containing element results
#
# with   relem.id                Element identifier
#        relem.nofpnt            Number of result points
#        relem.coor(2)           Coordinates of results points
#        relem.rtype             Result type: 3 = plane stress
#        relem.results.sig(3, :) Stresses: x, y, xy
#        relem.results.eps(3, :) Strains:  x, y, xy
#
# The function computes the results of 2-dimensional t3 elements.
#
# The variable argument list contains additional arguments depending
# on the class:
#
#    statresp:    none
#    freqresp:    varargin{1}      Circular excitation frequency
#
# -------------------------------------------------------------------

# Loop over elements

  for ne = 1 : nofelt

      ue = disp(elem(ne).ects, :);

      x = elem(ne).coor(:, 1);
      y = elem(ne).coor(:, 2);

      E = mfs_emat2d(elem(ne).mat);

      Jmat = [ x(2) - x(1), y(2) - y(1);
               x(3) - x(1), y(3) - y(1)];

      JE   = det(Jmat);

      BE = [ y(2) - y(3), 0, y(3) - y(1), 0, y(1) - y(2), 0;
             0, x(3) - x(2), 0, x(1) - x(3), 0, x(2) - x(1);
             x(3) - x(2), y(2) - y(3), x(1) - x(3), y(3) - y(1), ...
             x(2) - x(1), y(1) - y(2)] / JE;

      coor = [mean(x), mean(y)];
      eps  = BE * ue;
      sig  = E * eps;

      results   = struct("sig", sig, "eps", eps);
      relem(ne) = struct("id", elem(ne).id, "nofpnt", 1,
                          "coor", coor, "rtype", 3, "results", results);

  end

end
