function relem = mfs_reselt(nofelt, elem, disp, class, varargin)

# usage: relem = mfs_reselt(nofelt, elem, disp, class, varargin)
#
# Input  nofelt     Number of elements 
#        elem(:)    Structure array with element data
#        disp(:, :) Displacement matrix
#        class      Response class
#
# Output relem(:)   Structure containing stress results
#
# with   relem.id                   Element identifier
#        relem.nofpnt               Number of result points
#        relem.coor(:, 2)           Coordinates of results points
#        relem.rtype                Result type: 3 = plane stress
#        relem.results.sig(3, :, :) Stresses: x, y, xy
#        relem.results.eps(3, :, :) Strains:  x, y, xy
#
# The function computes the results of t3r elements.  # The last
# index of sig and eps correponds to result points.
#
# The variable argument list contains additional arguments depending
# on the class:
#
#    statresp:    none
#    freqresp:    varargin{1}      Circular excitation frequency
#
# -------------------------------------------------------------------

  ib = [2, 3, 1];     % Indices of edge end nodes

# Stress points

  ng = 1;
  [r, s, w] = mfs_gausstr(ng);

# Loop over elements

  for nel = 1 : nofelt

      coor = elem(nel).coor;
      u    = disp(elem(nel).ects, :);
      E    = mfs_emat2d(elem(nel).mat);

      T    = mfs_rot2mid2d(coor);
      ue   = T * u;

      coor = [coor; 0.5 * (coor + coor(ib, :))];

#   Loop over stress points

      for k = 1 : ng
          NE = mfs_NE_t6(r(k), s(k));
          BE = mfs_BE_t6(r(k), s(k), coor(:, 1), coor(:, 2));
          csp(k, :) = NE * coor;
          eps(:, :, k) = BE * ue;
          sig(:, :, k) = E * eps(:, :, k);
      end

      results    = struct("sig", sig, "eps", eps);
      relem(nel) = struct("id", elem(nel).id, "nofpnt", ng,
                          "coor", csp, "rtype", 3, "results", results);

  end

end
