function relem = mfs_reselt(nofelt, elem, disp, class, varargin)

# usage: relem = mfs_reselt(nofelt, elem, disp, class, varargin)
#
# Input  nofelt      Number of elements 
#        elem(:)     Structure array with element data
#        disp(:, :)  Displacement matrix
#        class       Response class
#
# Output relem(:)    Structure array containing stress results
#
# with   relem.id                   Element identifier
#        relem.nofpnt               Number of result points
#        relem.coor(:, 2)           Coordinates of results points
#        relem.rtype                Result type: 3 = plane stress
#        relem.results.sig(3, :, :) Stresses: x, y, xy
#        relem.results.eps(3, :, :) Strains:  x, y, xy
#
# The function computes the results of t6 elements.  # The last index
# of sig and eps corresponds to result points.
#
# The variable argument list contains additional arguments depending
# on the class:
#
#    statresp:    none
#    freqresp:    varargin{1}      Circular excitation frequency
#
# -------------------------------------------------------------------

# Stress points

  ng = 1;
  [r, s, w] = mfs_gausstr(ng);

# Loop over elements

  for nel = 1 : nofelt

      ue = disp(elem(nel).ects, :);
      E  = mfs_emat2d(elem(nel).mat);

#   Loop over stress points

      for k = 1 : ng
          x  = elem(nel).coor(:, 1);
          y  = elem(nel).coor(:, 2);
          NE = mfs_NE_t6(r(k), s(k));
          BE = mfs_BE_t6(r(k), s(k), x, y);
          coor(k, :) = NE * [x, y];
          eps(:, :, k)  = BE * ue;
          sig(:, :, k)  = E * eps(:, :, k);
      end

      results    = struct("sig", sig, "eps", eps);
      relem(nel) = struct("id", elem(nel).id, "nofpnt", ng,
                          "coor", coor, "rtype", 3, "results", results);

  end

end
