function rc = mfs_chkelt(msg, elem, Lmin)

# usage: rc = mfs_chkelt(msg, elem, Lmin)
#
# Input  msg    File handle of message file
#        elem   Structure with element data
#        Lmin   Minimum element length
#
# Ouput  rc     Return code: 0 means no erros
#                            1 means erros
#
# The function performs basic checks of 3-dimensional beam elements.
#
# ----------------------------------------------------------------------------

# Initialize

  geom_fields = {"A", "Iy", "Iz", "Iyz", "IT", "v", "Q", "CE", "P"};
  rc  = 0;

# Loop over elements

  for el = elem

      geom = el.geom;
      rce = mfs_checkfields2(msg, geom, geom_fields(1 : 5), "Element", 
                             el.id);
      if (! rce)
         for k = [1, 2, 3, 5]
             if (getfield(geom, geom_fields{k}) <= 0)
                fprintf(msg, "*E* Element %5d: Illegal value of \"geom.", 
                        el.id);
                fprintf(msg, "%s\"\n", geom_fields{k});
                rce = 1;
              end
         end
         if (isfield(geom, "CE") && length(geom.CE) != 2)
            fprintf(msg, "*E* Element %5d: Bad definition of \"geom.CE\"\n", 
                    el.id);
            rce = 1;
         end
         if (isfield(geom, "P") && length(geom.P) != 2)
            fprintf(msg, "*E* Element %5d: Bad definition of \"geom.P\"\n", 
                    el.id);
            rce = 1;
         end
      end

      if (isfield(geom, "v"))
         if (length(geom.v) != 3)
            fprintf(msg, "*E* Element %5d: Bad definition of \"geom.v\"\n", 
                    el.id);
            rce = 1;
         end
         v = geom.v;
      elseif (isfield(geom, "Q"))
         if (length(geom.Q) != 3)
            fprintf(msg, "*E* Element %5d: Bad definition of \"geom.Q\"\n", 
                    el.id);
            rce = 1;
         end
         v = geom.Q - el.coor(1, :);
      else
         fprintf(msg, "*E* Element %5d: Element axes undefined\n", el.id);
         rce = 1;
      end

      rce = rce | mfs_checkfields3(msg, geom, geom_fields, "Element", el.id);

      if (! rce)
         d = el.coor(2, :) - el.coor(1, :);
         L = norm(d);
         if (L < Lmin)
            rce = 1;
            fprintf(msg, "*E* Beam %5d has zero length\n", el.id);
         end
         ex = d' / L;
         ez = v' - (v * ex) * ex;
         if (norm(ez) < Lmin)
            rce = 1;
            fprintf(msg, "*E* Beam %5d:", el.id);
            fprintf(msg, " Direction vector is nearly parallel to beam axis\n");
         end

         if (isfield(geom, "P"))
            offset = sqrt(geom.P * geom.P') / L;
            if (offset > 0.5)
               fprintf(msg, "*W* Beam %5d:", el.id);
               fprintf(msg, " Large offset may cause numerical problems");
               fprintf(msg, " (%6.2f %% of length)\n", 100 * offset); 
            end
         end

      end

      rce = rce | mfs_chkmat3d(msg, el.mat, el.id);

      rc = rc | rce;

  end

end
