function k = mfs_ke(elem)

# usage: k = mfs_ke(elem)
#
# Input  elem     Structure containing data of one element
#
# Output k        Element stiffness matrix
#
# The function computes the stiffness matrix of a 3-dimensional beam 
# element.
#
# -------------------------------------------------------------------

# Geometry

  if (isfield(elem.geom, "Q"))
     v = elem.geom.Q - elem.coor(1, :);
  else
     v = elem.geom.v;
  end

  d = elem.coor(2, :) - elem.coor(1, :);
  L = norm(d);

  ex = d' / L;
  ez = v' - (v * ex) * ex;
  Lz = norm(ez);
  ez = ez / Lz;
  ey = cross(ez, ex);

# Stiffness for extension

  Ke = elem.mat.E * elem.geom.A / L;

# Bending stiffness

  Eb    = elem.mat.E / L^3;
  IyL   = Eb * elem.geom.Iy * L;
  IyL2  = IyL * L;
  IzL   = Eb * elem.geom.Iz * L;
  IzL2  = IzL * L;
  IyzL  = Eb * elem.geom.Iyz * L;
  IyzL2 = IyzL * L;

  k22  =  12 * Eb * elem.geom.Iz;
  k23  = -12 * Eb * elem.geom.Iyz;
  k25  =   6 * IyzL;
  k26  =   6 * IzL;
  k28  =  -k22;
  k29  =  -k23;

  k33  =  12 * Eb * elem.geom.Iy;
  k35  =  -6 * IyL;
  k36  =  -k25;
  k39  =  -k33;

  k55  =   4 * IyL2; 
  k56  =   4 * IyzL2;
  k59  =  -k35;
  k511 =   2 * IyL2;
  k512 =   2 * IyzL2;

  k66  =   4 * IzL2;
  k68  =  -k26;
  k612 =   2 * IzL2;

# Torsional stiffness

  G = elem.mat.E / (2 * (1 + elem.mat.ny));
  Kt = G * elem.geom.IT / L;

# Complete stiffness matrix in element coordinate system

%         u    v    w  thta   phi   psi   u    v    w  thta   phi   psi
  ke = [  Ke,   0,   0,   0,    0,    0, -Ke,   0,   0,   0,    0,    0;
           0, k22, k23,   0,  k25,  k26,   0, k28, k29,   0,  k25,  k26;
           0, k23, k33,   0,  k35,  k36,   0, k29, k39,   0,  k35,  k36;
           0,   0,   0,  Kt,    0,    0,   0,   0,   0, -Kt,    0,    0;
           0, k25, k35,   0,  k55,  k56,   0, k36, k59,   0, k511, k512;
           0, k26, k36,   0,  k56,  k66,   0, k68, k25,   0, k512, k612;
         -Ke,   0,   0,   0,    0,    0,  Ke,   0,   0,   0,    0,    0;
           0, k28, k29,   0,  k36,  k68,   0, k22, k23,   0,  k36,  k68;
           0, k29, k39,   0,  k59,  k25,   0, k23, k33,   0,  k59,  k25;
           0,   0,   0, -Kt,    0,    0,   0,   0,   0,  Kt,    0,    0;
           0, k25, k35,   0, k511, k512,   0, k36, k59,   0,  k55,  k56; 
           0, k26, k36,   0, k512, k612,   0, k68, k25,   0,  k56,  k66];

# Transformation matrix from structural to element system

  T33 = [ex'; ey'; ez'];
  T   = zeros(12, 12, "double");
  T( 1 :  3,  1 : 3) = T33;
  T( 4 :  6,  4 : 6) = T33;
  T( 7 :  9,  7 : 9) = T33;
  T(10 : 12, 10 :12) = T33;

  if (isfield(elem.geom, "CE"))
     TE  = eye(12, 12);
     TE(2,  4) = -elem.geom.CE(2);
     TE(3,  4) =  elem.geom.CE(1);
     TE(8, 10) =  TE(2, 4);
     TE(9, 10) =  TE(3, 4);
     T = TE * T;
  end

  if (isfield(elem.geom, "P"))
     TA = eye(12, 12);
     TA(1, 5) = -elem.geom.P(2);
     TA(1, 6) =  elem.geom.P(1);
     TA(2, 4) = -TA(1, 5);
     TA(3, 4) = -TA(1, 6);
     TA(7, 11) = -elem.geom.P(2);
     TA(7, 12) =  elem.geom.P(1);
     TA(8, 10) = -TA(7, 11);
     TA(9, 10) = -TA(7, 12);
     T = TA * T;
  end

# Transformation

  k = T' * ke * T;

end
